//=======================================================================================
// Flag Status Hud - by Mental Trousers
//=======================================================================================
//
// The first hud that I ever wrote was a direct rip off of the Madhud by Prodigy :) It
// was a straight forward ctf only, Flag Status hud. It has eventually evolved into this.
// This version is alot more efficient and flexible than that original one. It
// incorporates resolution handling, different positions on the screen, different layouts
// to use the screen positions, and a "slim" and "not so slim" mode layout. The "slim"
// layout has a total width of 120 pixels, while the "fat" mode is 220 pixels wide. The
// main advantage of the "fat" mode is that you get to see all of the name when someone
// grabs one of the flags. In slim mode the name may get chopped in half or something to
// fit into the field. However the "fat" mode takes up more screen real estate so is
// probably best suited to resolutions of 1024x768 or higher.
//
// Things covered in this script
// -----------------------------
// creating and manipulating objects from script
// resolution handling code
// more string matching etc
// using custom bitmaps on screen
// how to provide customisation (placement, look etc)
//
// This script only supports Ctf game play. There isn't any code in it to figure out
// which variant of the game is being played, because one of the servers that I play on
// alot uses a map rotation scheme that uses odd names for maps and doesn't set the
// variable that tells you if it's a ctf map or not.
//
// This script filters messages concerning the flags. It displays the name of the person
// who is carrying the flag (if someone has it), displays an icon that shows the status
// of the flag and keeps track of how many times each flag has been captured. The user is
// able to toggle the hud visible/invisible using the keybinding near the end of this
// block of comments or delete it from tribes altogether.
//
// The hud is contained within a single container called "FSH_container". This is just an
// object that can contain other objects and doesn't actually show up as anything on the
// screen, ie it's and invisible object. This is done so that when moving the hud around,
// only one thing needs to be moved ie the container. Also when toggling the visibility
// of the hud, all that needs to be done is to toggle the visibility of the container
// rather than all the objects it contains (even though you can't see the container, it
// can still be set to visible/invisible). Containers are all good. Use them everywhere.
//
// There are 2 other containers, "FSHOurFlagStatusContainer" and
// "FSHTheirFlagStatusContainer". These are used to chop the text displayed inside them
// so that the names of the flag carriers don't go outside the box they're in and cover
// up the icons.
//
// There are 3 types of objects used in this hud, the first is the container
// "SimGui::Control". As explained already, it's just a container. The second is the
// "FearGui::FearGuiMenu" object. This isn't actually a menu of any sort, rather it's the
// background that is often used for menus. It is that shaded box with the green partial
// border that you'll see on screen. All up there are 4 of these in this hud. The third
// object is the "FearGuiFormattedText" one. This is use to display text or bitmaps on
// the screen. The text can have some simple formatting performed on it, eg font,
// justification etc. Bitmaps can be displayed by enclosing the path within <>'s, but
// must be preceeded by "B#,#:" (the #'s are numbers)
// eg <B0,0:mt\\bmps\\flags\\flag_enemy.bmp>. The numbers are the offset from top left of
// the "FearGuiFormattedText" object 
//
// In order to use these objects to display what we want them to on the screen, we have
// to go through a couple of steps. The first is to actually create the objects. This is
// done using the "newObject" command. This will create the object for us, however it is
// created in the object tree root, so we have to move it onto the play screen. We do
// this using the "addToSet" command (a set is a container, just like the 3 that are
// created for the hud, or the PlayGui etc). Once it is added to the correct set, we have
// to give it some value to display. Using the "control::setValue" function is all that
// is required for the "FearGuiFormattedText" object. All objects default to visible when
// created so we don't need to do anythingelse for the "FearGui::FearGuiMenu" objects.
//
// After that, all we need to do is change the value of the "FearGuiFormattedText" as
// required to show what is happening with the flags.
//
// This hud also triggers a number of events concerning the flags. I use these events to
// trigger other scripts, for example I have one that taunts the opposition when we
// capture and says something like "You're REALLY lucky today ...." :)
//
//=======================================================================================

//purgeNGS clears out all traces of whatever you want from the Named Gui Set (ngs)
include ("mt\\purgeNGS");

//set the key bindings
EditActionMap("playMap.sae");
bindCommand(keyboard0, make, control, "/", TO, "FSH::toggleHud ();");
bindCommand(keyboard0, make, control, alt, "/", TO, "purgeNGSof (\"FSH\");");

//These two dictate the placement and slim/fat attributes. Placement is a number between
// 1 and 12 (1-5 are down the left, 6 is middle top, 7 is middle bottom, 8-12 are down
// the right), while $isSlim is true or false.
$placement=2;
$isSlim=true;

//=======================================================================================
// Sets up a bunch of vars that dictate the placement and the layout of the hud
// $hudPlacement::[vert, 1]     - vertical coord placement of hud container
// $hudPlacement::[horiz, 1]    - horizontal coord placement of hud container
// $hudConfig::[vert, #, bool]  - vertical coords for objects in hud container
// $hudConfig::[horiz, #, bool] - horizontal coords for objects in hud container
//
// The string of coords for the last 2 vars above are laid out according to the order in
// which the obhjects are created. See "function FSH::createHud (%gui)" for the order
//=======================================================================================
function FSH::handlePlacement ()
{
	//make sure it's a valid value
	if ($placement<1 || $placement>12)
		$placement=1;
	
	$hudPlacement::[vert, 1]=100;
	$hudPlacement::[vert, 2]=200;
	$hudPlacement::[vert, 3]=300;
	$hudPlacement::[vert, 4]=400;
	$hudPlacement::[vert, 5]=520;
	$hudPlacement::[vert, 6]=80;
	$hudPlacement::[vert, 7]=560;
	$hudPlacement::[vert, 8]=100;
	$hudPlacement::[vert, 9]=200;
	$hudPlacement::[vert, 10]=300;
	$hudPlacement::[vert, 11]=400;
	$hudPlacement::[vert, 12]=520;

	//If running in slim mode, the horizontal coords are different because the hud isn't
	//as wide as it is in fat mode. Also the labels are narrower.
	if (!$isSlim)
		{
		$hudConfig::oursTaken[false]="Ours (capped # times)";
		$hudConfig::theirsTaken[false]="Theirs (capped # times)";
		$hudConfig::[horiz, 1, false]="0 0 0 0 0 200 0 0 0 0 0 200";
		$hudConfig::[horiz, 2, false]="20 20 20 20 20 0 220 220 220 220 220 420";
		$hudConfig::[horiz, 3, false]="20 20 20 20 20 0 20 20 20 20 20 0";
		}
	else
		{
		$hudConfig::oursTaken[true]="Ours (scr: #)";
		$hudConfig::theirsTaken[true]="Them(scr: #)";
		$hudConfig::[horiz, 1, true]="0 0 0 0 0 100 0 0 0 0 0 100";
		$hudConfig::[horiz, 2, true]="20 20 20 20 20 0 120 120 120 120 120 220";
		$hudConfig::[horiz, 3, true]="20 20 20 20 20 0 20 20 20 20 20 0";
		}

	//Different placemets have different layouts. For instance, 1-5 have the text fields
	//stacked on top of each other with the icons on the right, while 8-12 are the same
	//except that the icons are on the left. 6 and 7 are split. The friendly icon is on
	//the left while the 2 text fields are stacked on top of each other. The unfriendly
	//is mirrored.
	if ($placement<=5)
		{
		$hudPlacement::[horiz, 1]=$hudPlacement::[horiz, 2]=$hudPlacement::[horiz, 3]=$hudPlacement::[horiz,  4]=$hudPlacement::[horiz,  5]=0;
		$hudConfig=1;
		$hudConfig::[vert, 1, false]=$hudConfig::[vert, 1, true]="0 0 20 20 0 20 40 40 60 60 0 60";
		}
	else if ($placement==6 || $placement==7)
		{
		//This simply adds 100 to the coords if running in slim mode. It takes advantage
		//of the fact that true=1 and false=0
		$hudPlacement::[horiz,  6]=$hudPlacement::[horiz,  7]=180+100*$isSlim;
		$hudConfig=2;
		$hudConfig::[vert, 2, false]=$hudConfig::[vert, 2, true]="0 0 20 20 0 20 0 0 20 20 0 20";
		}
	else
		{
		$hudPlacement::[horiz, 8]=$hudPlacement::[horiz, 9]=$hudPlacement::[horiz, 10]=$hudPlacement::[horiz, 11]=$hudPlacement::[horiz,  12]=580+100*$isSlim;
		$hudConfig=3;
		$hudConfig::[vert, 3, false]=$hudConfig::[vert, 3, true]="0 0 20 20 0 20 40 40 60 60 0 60";
		}
	return;
}


//=======================================================================================
// Adjusts the coordinates to suit the different resolutions.
//=======================================================================================
function FSH::handleResolution ()
{
	//the global var tells us what resolution the screen is
	%res=$pref::VideoFullScreenRes;

	//Once again, this relies on true=1 and false=0. Only one of the comparisons will
	//return true, all the rest will be zero. The coords are all set up for 800x600
	//(which is why there isn't a number for 800x600) and need to be adjusted for the
	//other resolutions
	$placementVert= $hudPlacement::[vert, $placement] * (0.585*(%res=="512x384") + 0.615*(%res=="640x400") + 0.765*(%res=="640x480") + (%res=="800x600") + 1.323*(%res=="1024x768"));

	//Once again, the horizontal coords are different if running in slim mode
	if ($isSlim)
		$placementHoriz= $hudPlacement::[horiz, $placement] * (0.577*(%res=="512x384") + 0.76*(%res=="640x400") + 0.764*(%res=="640x480") + (%res=="800x600") + 1.328*(%res=="1024x768"));
	else
		$placementHoriz= $hudPlacement::[horiz, $placement] * (0.505*(%res=="512x384") + 0.725*(%res=="640x400") + 0.725*(%res=="640x480") + (%res=="800x600") + 1.385*(%res=="1024x768"));
}

//***************************************************************************************
// Creates all of the objects and assembles them in the right order, puts the hud on
// screen and initialises it
//***************************************************************************************
function FSH::createHud (%gui)
{
	//only proceed if the playgui is opening. The $createHudOnce var is there because the
	//event that triggers this function wasn't detaching properly :(
	if (%gui!=playGui || !$createHudOnce)
		return;

	//make sure we don't create the hud again and detach from the event queue (hopefully)
	//also, remove any trace of an old hud from the object tree
	$createHudOnce=false;
	event::detach (eventGuiOpen, FSH::createHud, "%gui");
	purgeNGSof ("FSH");

	//figure out the adjustments for the various layouts and resolutions
	FSH::handlePlacement ();
	FSH::handleResolution ();

	//extract the coords for each object from the correct string. getWord() is space
	//delimited only, it treats commas etc as characters in a word
	for (%i=0;%i<12;%i++)
		{
		%horiz[%i]= getWord ($hudConfig::[horiz, $hudConfig, $isSlim], %i);
		%vert[%i]= getWord ($hudConfig::[vert, $hudConfig, $isSlim], %i);
		}

	//figure out the container size and size of the cells (horizontal size)
	%cont=440-200*$isSlim;
	%cell=200-100*$isSlim;

	//create all of the objects for the hud
	$hudContainer=newObject ("FSH_container", SimGui::Control, $placementHoriz, $placementVert, %cont, 400);

	$ourFlagLabelBG=newObject ("FSHOurFlagLabelBG", FearGui::FearGuiMenu, %horiz[0], %vert[0], %cell, 20);
	$ourFlagLabel=newObject ("FSHOurFlagLabel", FearGuiFormattedText, %horiz[1], %vert[1], %cell, 20);
	$ourFlagStatusBG=newObject ("FSHOurFlagStatusBG", FearGui::FearGuiMenu, %horiz[2], %vert[2], %cell, 20);
	$ourFlagStatusContainer=newObject ("FSHOurFlagStatusContainer", SimGui::Control, %horiz[3], %vert[3], %cell, 20);
	$ourFlagStatusText=newObject ("FSHOurFlagStatusText", FearGuiFormattedText, %horiz[4], %vert[4], %cell, 19);
	$ourFlagStatus=newObject ("FSHOurFlag", FearGuiFormattedText, %horiz[5], %vert[5], 20, 20);

	$theirFlagLabelBG=newObject ("FSHTheirFlagLabelBG", FearGui::FearGuiMenu, %horiz[6], %vert[6], %cell, 20);
	$theirFlagLabel=newObject ("FSHTheirFlagLabel", FearGuiFormattedText, %horiz[7], %vert[7], %cell, 20);
	$theirFlagStatusBG=newObject ("FSHTheirFlagStatusBG", FearGui::FearGuiMenu, %horiz[8], %vert[8], %cell, 20);
	$theirFlagStatusContainer=newObject ("FSHTheirFlagStatusContainer", SimGui::Control, %horiz[9], %vert[9], %cell, 20);
	$theirFlagStatusText=newObject ("FSHTheirFlagStatusText", FearGuiFormattedText, %horiz[10], %vert[10], %cell, 19);
	$theirFlagStatus=newObject ("FSHTheirFlag", FearGuiFormattedText, %horiz[11], %vert[11], 20, 20);

	//add all of the objects to the appropriate containers. The last one is the play gui
	addToSet ("FSH_container", $ourFlagLabelBG);
	addToSet ("FSH_container", $ourFlagLabel);
	addToSet ("FSH_container", $theirFlagLabelBG);
	addToSet ("FSH_container", $theirFlagLabel);
	addToSet ("FSH_container", $ourFlagStatusBG);
	addToSet ("FSH_container", $ourFlagStatus);
	addToSet ("FSH_container", $theirFlagStatusBG);
	addToSet ("FSH_container", $theirFlagStatus);
	addToSet ("FSHOurFlagStatusContainer", $ourFlagStatusText);
	addToSet ("FSHTheirFlagStatusContainer", $theirFlagStatusText);
	addToSet ("FSH_container", $ourFlagStatusContainer);
	addToSet ("FSH_container", $theirFlagStatusContainer);
	addToSet (PlayGui, $hudContainer);

	//set the initialisation values
	if($isSlim)
		$initVal=0;
	else
		$initVal="zero";
	control::setValue ("FSHOurFlagLabel","<jl><l2>"@string::replace ($hudConfig::oursTaken[$isSlim], "#", $initVal));
	control::setValue ("FSHOurFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_friendly.bmp>");
	control::setValue ("FSHTheirFlagLabel","<jl><l2>"@string::replace ($hudConfig::theirsTaken[$isSlim], "#", $initVal));
	control::setValue ("FSHTheirFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_enemy.bmp>");
}

//=======================================================================================
// Toggles the whether the hud is visible on screen or not
//=======================================================================================
function FSH::toggleHud ()
{
	%isVis=control::getVisible ("FSH_container");
	control::setVisible ("FSH_container", !%isVis);
}

//=======================================================================================
// When the server changes maps, we have to reset some of the variables and text etc
//=======================================================================================
function FSH::changeMission ()
{
	//reset the score count
	$FSH::weCapped=0;
	$FSH::theyCapped=0;

	//reset the text fields
	control::setValue ("FSHOurFlagLabel","<jl><l2>"@string::replace ($hudConfig::oursTaken[$isSlim], "#", $initVal));
	control::setValue ("FSHOurFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_friendly.bmp>");
	control::setValue ("FSHOurFlagStatusText", "");
	control::setValue ("FSHTheirFlagLabel","<jl><l2>"@string::replace ($hudConfig::theirsTaken[$isSlim], "#", $initVal));
	control::setValue ("FSHTheirFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_enemy.bmp>");
	control::setValue ("FSHTheirFlagStatusText", "");
}

//=======================================================================================
// This is where all of the messages are caught and filtered, with the correct objects
// update as needed. It's all relatively straight forward and easy to follow
//=======================================================================================
function FSH::clientMessages (%client, %msg)
{

	//if the word flag doesn't appear in the message then don't do anything
	if (match::string (%msg, "*"@"flag"@"*"))
		{

		//extract the name of the person that has the flag
		if (match::paramString (%msg, "%t took the%r"))
			{
			$taker = match::result (t);
			//with the formatted text object, the <>'s are special so we have to replace
			//them with somethingelse otherwise what is between them wont be displayed
			%taker = string::replace ($taker, "<", "{");
			$taker = string::replace (%taker, ">", "}");
			}

//TAKEN
		if(String::findSubStr(%msg, "our team has the") != -1)
			{
			event::trigger (eventEnemyHasFlag, $taker);
			control::setValue ("FSHTheirFlagStatusText"," <f2>"@$taker);
			control::setValue ("FSHTheirFlag","");
			}
		else if(String::findSubStr(%msg, "our team's flag has been taken.") != -1)	
			{
			event::trigger (eventHaveEnemyFlag, $taker);
			control::setValue ("FSHOurFlagStatusText"," <f2>"@$taker);
			control::setValue ("FSHOurFlag","");
			}
	
//CAPTURED
		else if(string::findsubstr(%msg, "our team captured the") != -1)
			{
			event::trigger (eventEnemyFlagCaptured);
			control::setValue ("FSHTheirFlagStatusText","");
			control::setValue ("FSHTheirFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_enemy.bmp>");
			$FSH::weCapped++;
			control::setValue ("FSHTheirFlagLabel","<jl><l2>"@string::replace ($hudConfig::theirsTaken[$isSlim], "#", $FSH::weCapped));
			}
		else if(string::findsubstr(%msg, "our team's flag was captured") != -1)
			{
			event::trigger (eventOurFlagCaptured);
			control::setValue ("FSHOurFlagStatusText","");
			control::setValue ("FSHOurFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_friendly.bmp>");
			$FSH::theyCapped++;
			control::setValue ("FSHOurFlagLabel","<jl><l2>"@string::replace ($hudConfig::oursTaken[$isSlim], "#", $FSH::theyCapped));
			}

//RETURNED
		else if(String::findSubStr(%msg, "Your flag was returned to base.") != -1)
			{
			event::trigger (eventOurFlagSafe);
			control::setValue ("FSHOurFlagStatusText","");
			control::setValue ("FSHOurFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_friendly.bmp>");
			}
		else if(String::findSubStr(%msg, "flag was returned to base.") != -1)
			{
			// They returned their flag
			event::trigger (eventEnemyFlagSafe);
			control::setValue ("FSHTheirFlagStatusText","");
			control::setValue ("FSHTheirFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_enemy.bmp>");
			}
	
//DROPPED	
		else if(String::findSubStr(%msg, "was dropped") != -1)
		{
		if(String::findSubStr(%msg, "your flag was dropped") != -1)
			{
			event::trigger (eventOurFlagDropped);
			control::setValue ("FSHOurFlagStatusText","<f2> ** Dropped **");
			control::setValue ("FSHOurFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_dropped.bmp>");
			}
		else
			{
			event::trigger (eventEnemyFlagDropped);
			control::setValue ("FSHTheirFlagStatusText","<f2> ** Dropped **");
			control::setValue ("FSHTheirFlag","<B0,0:mt\\bmps\\flags\\MT_Flag_dropped.bmp>");
			}
		}
	}
	//the message didn't contain anything to do with the flags
	else return;
}

//ensures that the hud is only created once
$createHudOnce=True;

//attach to the necessary events
event::attach (eventGuiOpen, FSH::createHud, %gui);
event::attach (eventChangeMission, FSH::changeMission);
event::attach (eventClientMessage, FSH::clientMessages, %client, %msg);
event::attach (eventExit, purgeNGSof, "FSH");