###################################################################
# Starsiege Terrain Exporter v1.2 2-22-05					#
# Copyright  2004 Super Powio						#
# Powio@hotmail.com								#
# www.SunAndShadows.com/MIB							#
#											#
# Special Thanks:									#
# ---------------									#
# Sentinal [MIB] - Beta Testing						#
# Raven [MIB] - Pointing out that I didn't need	a new exterior	#
# 	program									#
# 											#
# Michael Johnston (KineticPoet) - Original Tribes Script and	#
#	program (the one that's used here)					#
###################################################################

// This script and the included program are used in conjunction to make the 16-bit
// grayscale height maps that Unreal and some other games use for terrain. It does this
// by scanning the terrain and dumping raw heights into a text file called
// 'TerrainExporter.log'. From there you can run ConvTED.exe to interpret the data and
// create a 16-bit grayscale .RAW image file containing the height map.
//
// REQUIREMENTS: (Mostly taken from Michael Johnston's read-me file. Whups. ^_^ )
// 1.) Starsiege. Dur.
// 2.) ConvTED.exe, written by Michael "KineticPoet" Johnston. This is necessary for
//		converting the text dump to a raw binary file.
// 3.) A program that can open .RAW images as a 16-bit grayscale.
//
//	a) Adobe Photoshop.  Open your .raw file.  Set the import options to 16-bits,
//		1 channel, IBM-PC.
//
//	b) Daylon Leveller.  Import a raw binary file. Set options to 256x256 (or
//		whatever the header says in TerrainExport.log), unsigned 16-bit integers,
//		Intel, 0 byte header. I haven't tested this to see if it supports the extra
//		set of pixels from using the 'scan edges' argument in the script.
//
//		Leveller is available at: http://www.daylongraphics.com/products/leveller/
//
//	c) G16ed. This one is a height map editor for Unreal. Import a RAW image, 16 bits
//		per pixel, Intel, 0 byte header. This one isn't bad, but the current
//		version (1.45) can't resize images over 512 x 512 pixels, and only works in
//		powers of 2.
//
//		G16ed is available at: http://homepage.ntlworld.com/martingbell/
//
// INSTRUCTIONS:
// 1.  Copy this script to Starsiege's scripts folder, or its main directory.
// 2.  Delete TerrainExporter.log if it exists. You should always do this before
//	 exporting a new terrain or it will become quite messy.
// 3.  Start up Starsiege. If your console isn't already enabled, click the small white
//	 spot on the gold border of the main menu. It should be directly below the period
//	 of the version number in the lower left corner of the screen. Load the level
//	 whose terrain you want to export.
// 4.  Open up the console with the '~' key. If the console doesn't open up, see step
//	 #3. Type in 'exec("terrainExporter");' (without the quotes). You should get some
//	 information.
// 5.  Type in 'exportTerrain();' to begin exporting your terrain. This is assuming the
//	 default settings. You may add a few arguments as described in the
//	 ExportTerrainHelp() command (see step #4).
// 6.  Your game should seem to freeze up. Don't worry, this is normal. If you're using
//	 the default settings this should only take about 3 minutes. Otherwise, depending
//	 on the resolution of the height map you plan to export it could take upwards
//	 from 1 to 4 HOURS.
// 7.  Do NOT refresh your Starsiege folder. Do NOT open or even CLICK the newly
//	 generated TerrainExporter.log file while the script is still processing. You'll
//	 corrupt the file by doing so.
// 8.  When the script is done, you'll get an audio notification (unless Starsiege is
//	 running the background). Minimize/shut down the game, and look for
//	 'TerrainExporter.log' in your main Starsiege directory. Open it up using your
//	 favorite text editor.
// 9.  Remove the header and footer of the file (the lines that say BEGIN and END) so
//	 that you're left with just the numbers. Save that file in the same directory
//	 as ConvTED.exe.
// 10. Open up a DOS prompt, and navigate to where you extracted ConvTED.exe. Use the
//	 parameters: 'ConvTED.exe TerrainExporter.log myMap.raw'.
//	 This will convert the text dump into a binary file called myMap.raw.
// 11. Open the .raw file using one of the above mentioned programs.
// 12. Edit and/or save your image in a format that your BSP editor can read. It will
//	 most likely be a 16-bit grayscale in .BMP or .JPG format. If you created your
//	 height field using the 'scan edges' argument, you'll probably need to skim one
//	 pixel off right and bottom sides of the image (ex it's 257 x 257 instead of
//	 256 x 256).
// 13. Import your image into the BSP editor of your choice. Voila.
//
// USAGE:
// exportTerrain( [resolution], [areaParam], [heightParam], [scan edges] )
//		Scans the terrain and dumps a 16-bit height list into TerrainExporter.log
//
// ARGUMENTS:
//	resolution:	Changes the height map size in exponents of 2 (16, 32, 64, 128, etc)
//		Ranges from -4 - 3. Default is 0 (256).
//		-4 : 16 x 16
//		-3 : 32 x 32
//		-2 : 64 x 64
//		-1 : 128 x 128
//		 0 : 256 x 256
//		 1 : 512 x 512 (From here on, scanning takes a LONG time!)
//		 2 : 1024 x 1024
//		 3 : 2048 x 2048
//
//	areaParam:		Defines which area of the terrain you want to scan and export.
//		default	: Scans whatever's under the SAT map. Useful if the map has changed the SAT Map's position.
//		main		: Scans the main area of the terrain. This is the normal play area of the map.
//		secondary	: Scans the outer area of the terrain. This area is outside the normal play area, and automatically repeats itself
//					in a pattern around the map.
//		custom	: Scans a 8192m x 8192m area defined by the variables $TerrainExporter::CustomX and $TerrainExporter::CustomY.
//
//	heightParam:	Defines how the script should grab the minimum and maximum heights of the terrain.
//		default	: Scans the entire terrain.
//		area		: Scans the area being exported for the min and max height.
//		main		: Scans only the main area of the terrain for the min and max height.
//		secondary	: Scans only the secondary area of the terrain for the min and max height.
//		custom	: Uses the variables $TerrainExporter::CustomZMin and $TerrainExporter::CustomZMax to define the minimum and
//					maximum heights.
//
// VARIABLES:					
//
// $TerrainExporter::$CustomX	: Defines the upper left corner of the terrain to be exported. Works only when the areaParam argument
// $TerrainExporter::$CustomY	  is set to 'custom'.
// $TerrainExporter::$CustomZMin	: Sets the minimum and max heights of the terrain to be exported. Works only when the heightParam
// $TerrainExporter::$CustomZMax	  argument is set to 'custom'.
// $server::HudMapViewOffsetX		: These variables are built into the game. They change the center point of the SAT Map.
// $server::HudMapViewOffsetY
//
// TIPS:
// - When exporting the terrain, the game may seem to freeze. Don't panic, this is
//	normal. It can take hours to export the terrain at a higher resolutions.
// - Sometimes the SAT Map will not always be in the center of the map, for some maps.
//	When you're going to grab the main area of the terrain, make sure you use the
//	'main' option for the 'section' argument.
// - Be careful when setting a custom area. The edges of the terrain are 12288m away
//	to either side of the map. If the script scans a coordinate beyond the edge of
//	the map, you'll get an incorrect height map.
//
// ABOUTE THE 'SCAN EDGES' ARGUMENT:
// I realize that Starsiege has exactly 256 vertices on the main area of the terrain,
// however if you look at the SAT map, it's possible to count 257 vertices from upper-
// left corner to lower-right corner. This creates 32m worth of terrain between the
// edge of the SAT map and the main part of the terrain. The thing is: without the
// scan edges argument the script will cut off these 32m, and you may be left with
// an inaccurate heightfield (although only slightly! ;) ). If you want to fix this,
// simply set the scan edges argument to true. The script will generate a height field
// with one extra pixel, so that when you resize your image you will retain those 32m
// that the script would have otherwise cut off.
//
// However, sometimes this isn't really required at all. Since some BSP editors auto-
// matically cut off a small portion of terrain at the edge (such as Unreal) you may
// not want to set this argument.
//-----------------------------------------

function TerrainExporterHelp()
{
	echo("exportTerrain( [resolution], [areaParam], [heightParam], [scan edges] )");
	echo("resolution: Changes size of output image in exponenets of 2. Ranges as an integer from -4 (16x16) to 3 (2048x2048). Default 0 (256x256).");
	echo("areaParam: Selects the area of the terrain to export. See ReadMe.txt for more details.");
	echo("heightParam: Defines how the exporter will select the minimum and maximum heights. See ReadMe.txt for more details.");
	echo("scanEdges: Includes the right and lower edge of the scanned terrain into the output image. True/False.");
	echo("");
	echo("'$TerrainExporter::$CustomX' and '$TerrainExporter::$CustomY' define the upper left corner of the terrain to export when using a custom area.");
	echo("'$TerrainExporter::$CustomZMin' and '$TerrainExporter::$CustomZMax' define the custom height range.");
	echo("");
	echo("Please note that no arguments are actually required for execution, but they're there for fine tuning.");
	echo("Type TerrainExporterHelp(); to see this dialog again.");
}

function exportTerrain( %resolution, %areaParam, %heightParam, %edges )
{
	%resolution = floor(%resolution);	// Default 0 if not numeric

	// User does not input a valid resolution
	if( %resolution < -4 || %resolution > 3 )
	{
		echo( "exportTerrain: resolution must be between -4 and 3 (default:0)!" );
		return;
	}

	TerrainExporter::initRanges(%resolution, %areaParam, %edges);
	TerrainExporter::initZRange(%heightParam, %edges);

	// Check to see if the Z-Ranges are valid before exporting the terrain
	if($TerrainExporter::zMax <= $TerrainExporter::zMin)
	{
		echo("ERROR: zMax is less than or equal to zMin. Aborting...");
		return;
	}

	// Add a little more info to the log information string
	$TerrainExporter::LogInfo = strCat($TerrainExporter::LogInfo, "(Z-Min:"@ $TerrainExporter::zMin @" Z-Max:"@ $TerrainExporter::zMax @") " );

	// Okay, we're ready. Let's go.
	TerrainExporter::scanTerrain();
}

function TerrainExporter::initRanges( %resolution, %areaParam, %edges )
{
	// Calculate the dimensions of the output height map based on resolution input
	%mapSize = 8192;
	%heightMapSize = 256 * raiseToPower(2,%resolution);

	// Calculate the distance between points the terrain will be scanned at
	$TerrainExporter::increment = %mapSize / %heightMapSize;

	// The terrain is divided into two distinct sections; set the area to export
	if(%areaParam == main)
	{
		$TerrainExporter::xStart = -4096;
		$TerrainExporter::yStart = 4096;
	}
	else if(%areaParam == secondary)
	{
		$TerrainExporter::xStart = -4096;
		$TerrainExporter::yStart = 12288;
	}
	else if(%areaParam == custom)
	{
		$TerrainExporter::xStart = floor($TerrainExporter::CustomX);
		$TerrainExporter::yStart = floor($TerrainExporter::CustomY);
	}
	else	// Default: Satellite Map
	{
		%areaParam = Satellite;

		$TerrainExporter::xStart = floor($server::HudMapViewOffsetX - 4096);
		$TerrainExporter::yStart = floor($server::HudMapViewOffsetY + 4096);
	}
		$TerrainExporter::xEnd = $TerrainExporter::xStart + %mapSize;
		$TerrainExporter::yEnd = $TerrainExporter::yStart - %mapSize;

	// If %edges is true, the exporter will add one to the dimensions of the height map
	if(%edges)
	{
		$TerrainExporter::xEnd += $TerrainExporter::increment;
		$TerrainExporter::yEnd -= $TerrainExporter::increment;
		%heightMapSize = strCat(%heightMapSize, "+1");
	}

	// Set up the log information
	$TerrainExporter::LogInfo = $MissionName @" ("@ %areaParam @") ["@ %heightMapSize @"x"@ %heightMapSize @"] ";
}

function TerrainExporter::initZRange(%zParam, %edges)
{
	// Reset the Z Ranges
	$TerrainExporter::zMin = "";
	$TerrainExporter::zMax = "";

	// If the user has entered Z-Parameters for the task, skip straight to exporting
	if(%zParam == custom)
	{
		$TerrainExporter::zMin = floor($TerrainExporter::CustomZMin);
		$TerrainExporter::zMax = floor($TerrainExporter::CustomZMax);
		return;
	}
	else if(%zParam == area)
	{
		%xScanStart = $TerrainExporter::xStart;
		%yScanStart = $TerrainExporter::yStart;

		%xScanEnd = $TerrainExporter::xEnd;
		%yScanEnd = $TerrainExporter::yEnd;
	}
	else if(%zParam == main)
	{
		%xScanStart = -4096;
		%yScanStart = 4096;

		%xScanEnd = 4096;
		%yScanEnd = -4096;
	}
	else if(%zParam == secondary)
	{
		%xScanStart = -4096;
		%yScanStart = 12288;

		%xScanEnd = -4096;
		%yScanEnd = 4096;
	}
	else		// Default: main and secondary sections
	{
		%xScanStart = -4096;
		%yScanStart = 12288;

		%xScanEnd = 4096;
		%yScanEnd = -4096;
	}

	// Now scan for the Min and Max heights
	for(%y = %yScanStart; %y > %yScanEnd; %y -= $TerrainExporter::increment)
	{
		for(%x = %xScanStart; %x < %xScanEnd; %x += $TerrainExporter::increment)
		{
			if(%x == 12288)	// Point 12288 on the terrain is considered empty space, therefore we compensate
				%x -= 0.01;
			if(%y == 12288)
				%y -= 0.01;

			%height = getTerrainHeight(%x,%y);

			if(%height < $TerrainExporter::zMin || $TerrainExporter::zMin == "")
				$TerrainExporter::zMin = %height;
			if(%height > $TerrainExporter::zMax || $TerrainExporter::zMax == "")
				$TerrainExporter::zMax = %height;

			%x = round(%x);	// Starsiege sucks at handling decimals. This is after scanning buggy point 12288.
			%y = round(%y);
		}
	}
}

function TerrainExporter::scanTerrain()
{
	fileWrite( "TerrainExporter.log", append, strCat( "***BEGIN TERRAIN HEIGHTS*** ", $TerrainExporter::LogInfo, getTime()) );

	// The Scanner
	for(%y = $TerrainExporter::yStart; %y > $TerrainExporter::yEnd; %y -= $TerrainExporter::increment)
	{
		for(%x = $TerrainExporter::xStart; %x < $TerrainExporter::xEnd; %x += $TerrainExporter::increment)
		{
			if(%x == 12288)	// Point 12288 on the terrain is considered empty space, therefore we compensate
				%x -= 0.01;
			if(%y == 12288)
				%y -= 0.01;

			%height = getTerrainHeight(%x,%y);

			// The input ranges from 0 - 65535
			%input = round( ((%height - $TerrainExporter::zMin) / ($TerrainExporter::zMax - $TerrainExporter::zMin)) * 65535);

			if(%input > 65535)
				%input = 65535;
			if(%input < 0)
			{
				%input = 0;
			}

			// This becomes the output
			fileWrite( "TerrainExporter.log", append, %input );

			%x = round(%x);	// Starsiege sucks at handling decimals. This is after scanning buggy point 12288.
			%y = round(%y);
		}
	} // End of loop

	echo( "Terrain successfully exported to text file." );

	// Cool sound effects
	playSound( 0, "bptSlct.wav", IDPRF_2D);
	schedule( "playSound( 0, \"dlink_complete.wav\", IDPRF_2D);" , 0.5);

	fileWrite( "TerrainExporter.log", append, strCat( "***END TERRAIN HEIGHTS*** ", $TerrainExporter::LogInfo, getTime()) );
}

// Taken from TrigonometryStdLib.cs
function raiseToPower(%number, %exponent)
{
   if(isInteger(%exponent) == false)
   {
      echo("raiseToPower() - Exponents must be integers!");
      %exponent = round(%exponent);
      echo("Exponent rounded to " @ %exponent);
   }
   %answer = %number;
   if(%exponent == 0)
   {
      %answer = 1;
   }
   else if(%exponent == 1)
   {
      %answer = %number;
   }
   else if(%exponent < 0)
   {
      if(%exponent == -1)
      {
         %answer = (1 / %number);
      }
      else if(%exponent <= -2)
      {
         %count = -2;
         while(%count >= %exponent)
         {
            %count--;
            %answer = (%answer * %number); 
         }
         %answer = (1 / %answer);
      }
   }
   else if(%exponent >= 2)
   {
      %count = 2;
      while(%count <= %exponent)
      {
         %count++;
         %answer = (%answer * %number); 
      }
   }
   return %answer;
}

function isInteger(%number)
{
   %excess = (%number - floor(%number));
   if(%excess == 0)
   {
      return true;
   }
   else
   {
      return false;
   }
}

function round(%number)
{
   %roundNum = %number;
   if(%number > 0)
   {
      %excess = (%number - floor(%number));
      if(%excess >= 0.5)
      {
         %roundNum = (floor(%number) + 1);
      }
      else
      {
         %roundNum = floor(%number);
      }
   }
   if(%number < 0)
   {
      %excess = (floor(%number) - %number);
      if(%excess <= 0.5)
      {
         %roundNum = floor(%number);
      }
      else
      {
         %roundNum = (floor(%number) - 1);
      }
   }
   return %roundNum;
}

TerrainExporterHelp();