// --------------------------
// FishHUD v1.2 by SuckerFish
// --------------------------
//
// Introduction:
//
// FishHUD is essentially a major rewrite of DynHUD, written by KillerBunny.
// I noticed that DynHUD was missing kills occasionally, so I decided to take
// a closer look at the script.
//
// I won't go into detail as to the bugs I found in the DynHUD code, but
// suffice it to say that I replaced all the kill message parsing code and
// added some stuff to the flag event parsing code.
//
// Version history:
//
// 1.2: 3/22/99
//	- wasn't parsing the message "PLAYER adds VICTIM to her list of
//        sniper victims."; bug fixed
//
// 1.1: 3/21/99
//      - if friendly flag was dropped in the field and returned on its own,
//        the message, "Your flag was returned to base." was not processed;
//        bug fixed
//
// 1.0: 3/18/99
//      - file created
//
// Description:
//
// At the top of the HUD, there are two lines, one starting with an "E:" and
// one with an "F:". These two lines show the current status of the enemy
// and friendly flags, respectively. If the flag is taken, the appropriate
// line will show the name of the flag carrier; if the flag is dropped in the
// field, the flag status will simply read "Field".
//
// Below the two flag status lines is a bitmap of a flag. At a glance, it
// shows the status of your flag, green for safe, red if your flag was taken
// or dropped in the field.
//
// Next to the flag bitmap is a count of kills, mines, grenades, and repair
// kit. With the exception of the kill counter which is updated whenever you
// kill someone, these counts are updated once every two seconds. Note that
// if you change teams, these numbers will all be reset.
//
// The HUD is toggled on/off using CTRL-H, but you can change the key binding
// below.
//
// Usage:
//
// 1) Place this file in your Tribes /config/ directory.
// 2) Add the following line to the autoexec.cs file in your tribes /base/
//    directory:
//
//    exec("fish_hud.cs");
//
//    If you do not have an autoexec.cs file, you'll have to create one.
//
// 3) If you already using an onClientMessage() function, add the following
//    line to the top of it:
//
//    FishHUD::Parse(%client, %msg);
//
//    Otherwise, uncomment the onClientMessage() function at the bottom of
//    this script by removing the double-slashes ("//") in front of each
//    line.
//
// Script limitations:
//
// 1) FishHUD only works on unmodified servers. The HUD may not update
//    properly if there are different flag and kill messages.
// 2) FishHUD was designed to work for two-team CTF missions. For other types
//    of missions, the kill counter should still work; the flag info will
//    simply not update.
// 3) If you join a game where one or both flags are already taken, the HUD
//    won't show the correct status of the flags. The HUD will only be
//    updated with correct information the next time flag status changes.
// 4) The string display functions in Tribes use formatting tags similar to
//    those in HTML. If a player's name just happens to contain one of these
//    formatting tags, the name may not be displayed properly.
//
// Other notes:
//
// Please excuse the piss-poor coding style. I had a devil of a time figuring
// out how the scripting language works and it doesn't help that there is no
// official script documentation (yet) or debugging tools. I'm sure there are
// some things I could have written more elegantly, but I only know a
// miniscule fraction of the functions that are available in the language.
//
// Thanks and Acknowledgements go to KillerBunny and GotMilk? for the
// original DynHUD script, which FishHUD is based on. Thanks also to SeaBee
// for finding the "Your flag was returned to base." bug.
//
// See you online!
//                 - SuckerFish, 3/22/99

// ===========
// Key Binding
// ===========

editActionMap("playMap.sae");
bindCommand(keyboard0, make, control, "h", TO, "FishHUD::Toggle();");
bindCommand(keyboard0, break, control, "h", TO, "");

// ================
// String functions
// ================

// gets the length of a string
function String::len(%string) 
{
	for (%i = 0; String::getSubStr(%string, %i, 1) != ""; %i++)
		%length++;

	return %length;
}

// finds the last occurence of %substring in %text
function String::findLastSubStr(%text, %substring)
{
	%index = String::findSubStr(%text, %substring);
	if (%index < 0)
		return %index;

	%substring_length = String::len(%substring);
	%temp_string = %text;
	%last_index = %index;

	while (%index >= 0)
	{
		%num_chars = String::len(%temp_string) - %index - $substring_length;
		%temp_string = String::getSubStr(%temp_string, %index + %substring_length, %num_chars);
		%index = String::findSubStr(%temp_string, %substring);
		if (%index >= 0)
			%last_index += %substring_length + %index;
	}

	return %last_index;
}

// ======================================
// FishHUD routines
// ======================================

$FishHUD::Name = "FishHUD v1.2";

function FishHUD::Create()
{
	if($FishHUD::Loaded == 1)
		return;

	$FishHUD::Loaded = 1;

	$FishHUD[0] = newObject("FishHUD_Frame", FearGui::FearGuiMenu, 6, 85, 120, 108);
	$FishHUD[1] = newObject("FishHUD_Main", FearGuiFormattedText, 9, 82, 117, 105);

	for(%i = 0; $FishHUD[%i] != ""; %i++)
		addToSet(PlayGui, $FishHUD[%i]); 

	FishHUD::Heartbeat();
}

function FishHUD::UpdateHUD()
{
	if ($FishHUD::Loaded != 1)
		return;

	if ($FishHUD::EnemyCarrier != "")
		$FishHUD::FlagStatus = "<Bflag_enemycaptured.bmp>";
	else
		$FishHUD::FlagStatus = "<Bflag_atbase.bmp>";

	if ($FishHUD::FriendlyCarrier == "")
		$FishHUD::CarrierStatus = "<f3>E: Safe";
	else if ($FishHUD::FriendlyCarrier == "DROPPEDINTHEFIELD")
		$FishHUD::CarrierStatus = "<f3>E: <f1>Field";
	else
                $FishHUD::CarrierStatus = "<f3>E: <f2>" @ $FishHUD::FriendlyCarrier;

	if ($FishHUD::EnemyCarrier == "")
		$FishHUD::CarrierStatus = $FishHUD::CarrierStatus @ "\n<f3>F: Safe";
	else if ($FishHUD::EnemyCarrier == "DROPPEDINTHEFIELD")
		$FishHUD::CarrierStatus = $FishHUD::CarrierStatus @ "\n<f3>F: <f1>Field";
	else
                $FishHUD::CarrierStatus = $FishHUD::CarrierStatus @ "\n<f3>F: <f2>" @ $FishHUD::EnemyCarrier;

        if($FishHUD::KillCount == "")
                $FishHUD::KillCount = 0;

        if($FishHUD::MineCount == "")
                $FishHUD::MineCount = 0;

        if($FishHUD::GrenadeCount == "")
                $FishHUD::GrenadeCount = 0;

        if($FishHUD::RepairKitCount == "")
                $FishHUD::RepairKitCount = 0;

        $FishHUD::StatusString = "<f2>" @ $FishHUD::Name @ "\n<jl>" @ $FishHUD::CarrierStatus @ "\n\n" @ $FishHUD::FlagStatus @ "<L9><f3>Kills: <f2>" @ $FishHUD::KillCount @ "\n<f3>Mines:<f2> " @ $FishHUD::MineCount @ "\n<f3>Grens: <f2>" @ $FishHUD::GrenadeCount @ "\n<f3>Repair: <f2>" @ $FishHUD::RepairKitCount;

        Control::setValue("FishHUD_Main", $FishHUD::StatusString);
}

function FishHUD::Heartbeat()
{
        if($FishHUD::Loaded == 0)
		return;

	$FishHUD::MineCount = getItemCount("Mine");
	$FishHUD::GrenadeCount = getItemCount("Grenade");
	$FishHUD::RepairKitCount = getItemCount("Repair Kit");

        schedule("FishHUD::Heartbeat();", 2);
        FishHUD::UpdateHUD();
}

function FishHUD::Remove()
{
        if($FishHUD::Loaded == 0)
		return;

        $FishHUD::Loaded = 0;
        for(%i = 0; $FishHUD[%i] != ""; %i++)
                deleteObject($FishHUD[%i]);
}

function FishHUD::Toggle()
{
        if($FishHUD::Loaded == 1)
                FishHUD::Remove();
	else
                FishHUD::Create();
}

function FishHUD::Reset()
{
	// initialize HUD variables
	$FishHUD::EnemyCarrier = "";
	$FishHUD::FriendlyCarrier = "";
	$FishHUD::StatusString = "";
	$FishHUD::KillCount = 0;
	$FishHUD::MineCount = 0;
	$FishHUD::GrenadeCount = 0;
	$FishHUD::RepairKitCount = 0;

	// "VICTIM succumbs to PLAYER's rain of blaster fire."
	$PreVictimText[0]	= "";
	$PostVictimText[0]	= " succumbs to " @ $PCFG::Name @ "'s rain of blaster fire.";
        $Guns[0]		= "Blaster";

	// "VICTIM gets a blast out of PLAYER."
	$PreVictimText[1]	= "";
	$PostVictimText[1]	= " gets a blast out of " @ $PCFG::Name @ ".";
        $Guns[1]		= "Blaster";

	// "PLAYER's puny blaster shows VICTIM a new world of pain"
	$PreVictimText[2]	= $PCFG::Name @ "'s puny blaster shows ";
	$PostVictimText[2]	= " a new world of pain.";
        $Guns[2]              = "Blaster";

	// "VICTIM meets PLAYER's master blaster."
	$PreVictimText[3]	= "";
	$PostVictimText[3]	= " meets " @ $PCFG::Name @ "'s master blaster.";
        $Guns[3]		= "Blaster";

	// "PLAYER gives VICTIM a white-hot plasma injection."
	$PreVictimText[4]	= $PCFG::Name @ " gives ";
	$PostVictimText[4]	= " a white-hot plasma injection.";
        $Guns[4]		= "Plasma Gun";

	// "PLAYER asks VICTIM, 'Got plasma?'"
	$PreVictimText[5]	= $PCFG::Name @ " asks ";
	$PostVictimText[5]	= ", 'Got plasma?'";
        $Guns[5]		= "Plasma Gun";

	// "PLAYER gives VICTIM a plasma transfusion."
	$PreVictimText[6]	= $PCFG::Name @ " gives ";
	$PostVictimText[6]	= " a plasma transfusion.";
        $Guns[6]		= "Plasma Gun";

	// "VICTIM feels the warm glow of PLAYER's plasma."
	$PreVictimText[7]	= "";
	$PostVictimText[7]	= " feels the warm glow of " @ $PCFG::Name @ "'s plasma.";
        $Guns[7]		= "Plasma Gun";

	// "PLAYER ventilates VICTIM with his chaingun."
	$PreVictimText[8]	= $PCFG::Name @ " ventilates ";
	$PostVictimText[8]	= " with his chaingun.";
        $Guns[8]		= "Chaingun";

	// "PLAYER ventilates VICTIM with her chaingun."
	$PreVictimText[9]	= $PCFG::Name @ " ventilates ";
	$PostVictimText[9]	= " with her chaingun.";
        $Guns[9]		= "Chaingun";

	// "PLAYER guns down VICTIM."
	$PreVictimText[10]	= $PCFG::Name @ " guns down ";
	$PostVictimText[10]	= ".";
        $Guns[10]		= "Chaingun";

	// "PLAYER gives VICTIM an overdose of lead."
	$PreVictimText[11]	= $PCFG::Name @ " gives ";
	$PostVictimText[11]	= " an overdose of lead.";
        $Guns[11]		= "Chaingun";

	// "PLAYER fills VICTIM full of holes."
	$PreVictimText[12]	= $PCFG::Name @ " fills ";
	$PostVictimText[12]	= " full of holes.";
        $Guns[12]		= "Chaingun";

	// "PLAYER blasts VICTIM with a well-placed disc."
	$PreVictimText[13]	= $PCFG::Name @ " blasts ";
	$PostVictimText[13]	= " with a well-placed disc.";
        $Guns[13]		= "Disk Launcher";

	// "VICTIM catches a Frisbee of Death thrown by PLAYER."
	$PreVictimText[14]	= "";
	$PostVictimText[14]	= " catches a Frisbee of Death thrown by " @ $PCFG::Name @ ".";
        $Guns[14]		= "Disk Launcher";

	// "PLAYER's spinfusor caught VICTIM by surprise."
	$PreVictimText[15]	= $PCFG::Name @ "'s spinfusor caught ";
	$PostVictimText[15]	= " by surprise.";
        $Guns[15]		= "Disk Launcher";

	// "VICTIM falls victim to PLAYER's Stormhammer."
	$PreVictimText[16]	= "";
	$PostVictimText[16]	= " falls victim to " @ $PCFG::Name @ "'s Stormhammer.";
        $Guns[16]		= "Disk Launcher";

	// "PLAYER blows VICTIM up real good."
	$PreVictimText[17]	= $PCFG::Name @ " blows ";
	$PostVictimText[17]	= " up real good.";
        $Guns[17]		= "Grenade Launcher / Grenade / Mine";

	// "VICTIM never saw it coming from PLAYER."
	$PreVictimText[18]	= "";
	$PostVictimText[18]	= " never saw it coming from " @ $PCFG::Name @ ".";
        $Guns[18]		= "Grenade Launcher / Grenade / Mine";

	// "VICTIM gets a taste of PLAYER's explosive temper."
	$PreVictimText[19]	= "";
	$PostVictimText[19]	= " gets a taste of " @ $PCFG::Name @ "'s explosive temper.";
        $Guns[19]		= "Grenade Launcher / Grenade / Mine";

	// "PLAYER gives VICTIM a fatal concussion."
	$PreVictimText[20]	= $PCFG::Name @ " gives ";
	$PostVictimText[20]	= " a fatal concussion.";
        $Guns[20]		= "Grenade Launcher / Grenade / Mine";

	// "VICTIM becomes a victim of PLAYER's laser rifle."
	$PreVictimText[21]	= "";
	$PostVictimText[21]	= " becomes a victim of " @ $PCFG::Name @ "'s laser rifle.";
        $Guns[21]		= "Laser Rifle";

	// "PLAYER adds VICTIM to his list of sniper victims."
	$PreVictimText[22]	= $PCFG::Name @ " adds ";
	$PostVictimText[22]	= " to his list of sniper victims.";
        $Guns[22]		= "Laser Rifle";

	// "PLAYER adds VICTIM to her list of sniper victims."
	$PreVictimText[23]	= $PCFG::Name @ " adds ";
	$PostVictimText[23]	= " to her list of sniper victims.";
        $Guns[23]		= "Laser Rifle";

	// "PLAYER fells VICTIM with a sniper shot."
	$PreVictimText[24]	= $PCFG::Name @ " fells ";
	$PostVictimText[24]	= " with a sniper shot.";
        $Guns[24]		= "Laser Rifle";

	// "VICTIM stayed in PLAYER's crosshairs for too long."
	$PreVictimText[25]	= "";
	$PostVictimText[25]	= " stayed in " @ $PCFG::Name @ "'s crosshairs for too long.";
        $Guns[25]		= "Laser Rifle";

	// "PLAYER short-circuits VICTIM's systems."
	$PreVictimText[26]	= $PCFG::Name @ " short-circuits ";
	$PostVictimText[26]	= "'s systems.";
        $Guns[26]		= "ELF Gun";

	// "VICTIM gets zapped with PLAYER's ELF gun."
	$PreVictimText[27]	= "";
	$PostVictimText[27]	= " gets zapped with " @ $PCFG::Name @ "'s ELF gun.";
        $Guns[27]		= "ELF Gun";

	// "VICTIM gets a real shock out of meeting PLAYER."
	$PreVictimText[28]	= "";
	$PostVictimText[28]	= " gets a real shock out of meeting " @ $PCFG::Name @ ".";
        $Guns[28]		= "ELF Gun";

	// "PLAYER gives VICTIM a nasty jolt."
	$PreVictimText[29]	= $PCFG::Name @ " gives ";
	$PostVictimText[29]	= " a nasty jolt.";
        $Guns[29]		= "ELF Gun";

	// "PLAYER mortars VICTIM into oblivion."
	$PreVictimText[30]	= $PCFG::Name @ " mortars ";
	$PostVictimText[30]	= "into oblivion.";
	$Guns[30]		= "Heavy Mortar";

	// "VICTIM didn't see that last mortar from PLAYER."
	$PreVictimText[31]	= "";
	$PostVictimText[31]	= " didn't see that last mortar from " @ $PCFG::Name @ ".";
	$Guns[31]		= "Heavy Mortar";

	// "PLAYER inflicts a mortal mortar wound on VICTIM."
	$PreVictimText[32]	= $PCFG::Name @ " inflicts a mortal mortar wound on ";
	$PostVictimText[32]	= ".";
	$Guns[32]		= "Heavy Mortar";

	// "PLAYER'S mortar takes out VICTIM."
	$PreVictimText[33]	= $PCFG::Name @ "'s mortar takes out ";
	$PostVictimText[33]	= ".";
	$Guns[33]		= "Heavy Mortar";
}

// ============================================
// FishHUD parsing routines
// ============================================

function FishHUD::Parse(%client, %msg)
{
	if(%client != 0)
		return;

	// look for player name in message
	%NameIndex = String::findSubStr(%msg, $PCFG::Name);

	if (%NameIndex >= 0)
	{
		// check if player changed teams
		if (FishHUD::teamEvents(%client, %msg))
		{
			FishHUD::UpdateHUD();
			return;
		}

		// check if player killed someone
		if (FishHUD::killEvents(%client, %msg))
		{
			FishHUD::UpdateHUD();
			return;
		}
	}

        FishHUD::flagEvents(%client, %msg);
        FishHUD::UpdateHUD();
}

function FishHUD::teamEvents(%client, %msg)
{
	if (%msg == $PCFG::Name @ " changed teams.")
	{
		%temp_carrier = $FishHUD::EnemyCarrier;
		$FishHUD::EnemyCarrier = $FishHUD::FriendlyCarrier;
		$FishHUD::FriendlyCarrier = %temp_carrier;

		$FishHUD::KillCount = 0;
		$FishHUD::MineCount = 0;
		$FishHUD::GrenadeCount = 0;
		$FishHUD::RepairKitCount = 0;

		return 1;
	}
	else
	{
		return 0;
	}
}

function FishHUD::killEvents(%client, %msg)
{
	// clear victim variable
	%victim = "";

	// for each kill message
	for (%i = 0; $Guns[%i] != ""; %i++)
	{
		// if the victim hasn't been identified yet
		if (%victim == "")
		{
			// find the locations of the pre- and post- victim text in the message
			%PreVictimIndex = String::findSubStr(%msg, $PreVictimText[%i]);
			%PostVictimIndex = String::findLastSubStr(%msg, $PostVictimText[%i]);

			// if the pre- and post- victim text appear in the message
			if (%PreVictimIndex == 0 && %PostVictimIndex >= 0) {
				%msg_len = String::len(%msg);
				%start = String::len($PreVictimText[%i]);
				%num_chars = %PostVictimIndex - %start;

				// get the victim
				%victim = String::getSubStr(%msg, %start, %num_chars);

				// get the murder weapon
				%murder_weapon = $Guns[%i];
			}
		}
	}

	// if a victim was found and the victim wasn't the player
	if (%victim != "" && %victim != $PCFG::Name)
	{
		// print kill message
		Client::centerPrint("<jc><f3>Kill Added\n<f0>Killed: <f1>" @ %victim @"\n <f0>Weapon: <f1>" @ %murder_weapon, 1);
		schedule("Client::centerPrint(\"\", 1);", 4);

		// increment kill count
		$FishHUD::KillCount++;

		return 1;
	} else {
		return 0;
	}
}

function FishHUD::flagEvents(%client, %msg)
{
	%posit = String::findSubStr(%msg, " took the ");

	if (%posit > 0)
	{
		$taker = String::getSubStr(%msg, 0, %posit);
		return;
	}
	
	if (String::findSubStr(%msg, "Your team has the ") == 0)
	{
		if($taker == "You")
                        $FishHUD::FriendlyCarrier = "You";
		else
		{     
			Client::centerPrint("<jc><f2>" @ $taker @ " has the enemy's flag.", 1); 
			schedule("Client::centerPrint(\"\", 1);", 5);
	
                        $FishHUD::FriendlyCarrier = $taker;
		}
	}
	else if (%msg == "Your team's flag has been taken.")
	{
		Client::centerPrint("<jc><f2>" @ $taker @ " has taken our flag!", 1); 
		schedule("Client::centerPrint(\"\", 1);", 5);

                $FishHUD::FlagSafe = 0;
                $FishHUD::EnemyCarrier = $taker;
	}
	else if (%msg == "Your team captured the flag.")
	{
                $FishHUD::FlagSafe = 1;
                $FishHUD::FriendlyCarrier = "";
                $FishHUD::EnemyCarrier = "";
	}
	else if (String::findSubStr(%msg, " captured the ") > 0)
	{
                $FishHUD::FlagSafe = 1;
                $FishHUD::FriendlyCarrier = "";
                $FishHUD::EnemyCarrier = "";
	}
	else if (String::findSubStr(%msg, "Your flag was returned to base.") == 0)
	{
                $FishHUD::FlagSafe = 1;
                $FishHUD::EnemyCarrier = "";
	}
	else if (String::findSubStr(%msg, " flag was returned to base.") > 0)
	{
                $FishHUD::FriendlyCarrier = "";
	}
	else if (%msg == "Your flag was dropped in the field.")
	{
		$FishHUD::EnemyCarrier = "DROPPEDINTHEFIELD";
	}
	else if (String::findSubStr(%msg, " flag was dropped in the field.") > 0)
	{
		$FishHUD::FriendlyCarrier = "DROPPEDINTHEFIELD";
	}
}

// ======================================
// Functions to OverRide to support HUD
// ======================================

function onExit()
{
	// Added by KB
        FishHUD::Remove();

	if(isObject(playGui))
		storeObject(playGui, "config\\play.gui");

	saveActionMap("config\\config.cs", "actionMap.sae", "playMap.sae", "pdaMap.sae");

	//update the video mode - since it can be changed with alt-enter
	$pref::VideoFullScreen = isFullScreenMode(MainWindow);

	checkMasterTranslation();
	echo("exporting pref::* to prefs.cs");
	export("pref::*", "config\\ClientPrefs.cs", False);
	export("Server::*", "config\\ServerPrefs.cs", False);
	export("pref::lastMission", "config\\ServerPrefs.cs", True);
	BanList::export("config\\banlist.cs");
}

function onConnection(%message)
{
	echo("Connection ", %message);
	$dataFinished = false;
	if(%message == "Accepted")
	{
		resetSimTime();

		//execute the custom script
		if ($PCFG::Script != "")
		{
			exec($PCFG::Script);
		}

		resetPlayDelegate();
		remoteEval(2048, "SetCLInfo", $PCFG::SkinBase, $PCFG::RealName, $PCFG::EMail, $PCFG::Tribe, $PCFG::URL, $PCFG::Info, $pref::autoWaypoint, $pref::noEnterInvStation, $pref::messageMask);

		if ($Pref::PlayGameMode == "JOIN")
		{
			cursorOn(MainWindow);
			GuiLoadContentCtrl(MainWindow, "gui\\Loading.gui");
			renderCanvas(MainWindow);
		}

		FishHUD::Remove();
                FishHUD::Reset();
	}
	else if(%message == "Rejected")
	{
		Quickstart();
		$errorString = "Connection to server rejected:\n" @ $errorString;
		GuiPushDialog(MainWindow, "gui\\MessageDialog.gui");
		schedule("Control::setValue(MessageDialogTextFormat, $errorString);", 0);
	}
	else
	{
		Quickstart();

		if(%message == "Dropped")
		{
			if($errorString == "")
				$errorString = "Connection to server lost:\nServer went down.";
			else
				$errorString = "Connection to server lost:\n" @ $errorString;

			GuiPushDialog(MainWindow, "gui\\MessageDialog.gui");
			schedule("Control::setValue(MessageDialogTextFormat, $errorString);", 0);
		}
		else if(%message == "TimedOut")
		{
			$errorString = "Connection to server timed out.";
			GuiPushDialog(MainWindow, "gui\\MessageDialog.gui");
			schedule("Control::setValue(MessageDialogTextFormat, $errorString);", 0);
		}
	}
}

function remoteMissionChangeNotify(%serverManagerId, %nextMission)
{
	if(%serverManagerId == 2048)
	{
                FishHUD::Reset();
                FishHUD::Remove();

		echo("Server mission complete - changing to mission: ", %nextMission);
		echo("Flushing Texture Cache");
		flushTextureCache();
		schedule("purgeResources(true);", 3);
	}
}

//function onClientMessage(%client, %msg)
//{
//	if(%client)
//		$lastClientMessage = %client;
//
//        FishHUD::Parse(%client, %msg);
//	
//	return true;
//}
