/*
 
   $Workfile:   bitStream.h  $
 
   DESCRIPTION:
 
 
   (c) Copyright 1996, Dynamix Inc.   All rights reserved.
 
  */

#ifndef __BITSTREAM_H__
#define __BITSTREAM_H__

#include "T1.h"

/*
    - BitStream class for T1
	- mCharFreq table has +1 to all fields and +1 to isalpha & isdigit 
	  embedded (Unique to Tribes 1)
*/

class BitStream;

class HuffmanProcessor {
	struct HuffNode {
		U32 pop;
		S32 index[2];
		U32 pad;
	};

	struct HuffLeaf {
		U32 pop;
		U8  numBits;
		U8  symbol;
		U32 code;
		U32 pad;
	};

	struct HuffWrap {
		HuffNode* pNode;
		HuffLeaf* pLeaf;

	public:
		HuffWrap(                   ) : pNode( NULL ), pLeaf( NULL ) { }
		void set( HuffLeaf* in_leaf ) { pNode = NULL; pLeaf = in_leaf; }
		void set( HuffNode* in_node ) { pLeaf = NULL; pNode = in_node; }

		U32 getPop() { if ( pNode ) return ( pNode->pop ); else return ( pLeaf->pop ); }
	};

	void buildTables( );
	S16 determineIndex( HuffWrap& );
	void generateCodes( BitStream &rBS, S32 index, S32 depth );

	bool          mTablesBuilt;
	static const  U32 mCharFreqs[ 256 ];
	
	S16           mHuffNodeCount;
	HuffNode      mHuffNodes [ 256 ];
	HuffLeaf      mHuffLeaves[ 256 ];


public:
	HuffmanProcessor( ) : mTablesBuilt( false ) { }
	void readHuffBuffer( BitStream &stream, char *buffer );
	bool writeHuffBuffer( BitStream &stream, const char *buffer, S32 maxLen );

	static HuffmanProcessor gHuffProcessor;
};

/*
	Bitstream class
*/

class BitStream {
private:	
	S32 mBitNum;
	bool mError;
	S32 mBufSize;
	U8 *mDataPtr;
	S32 mMaxBitNum;

	friend class HuffmanProcessor;

public:
	BitStream( ) : mError(true), mBufSize( 0 ), mDataPtr( NULL ) {} 
	BitStream( void *bufPtr, S32 bufSize ) { setBuffer( bufPtr, bufSize ); }

	void burnBits( S32 nbits );

	U8 *getBuffer( ) const { return ( mDataPtr ); }
	U8 *getCurBuffer( ) const { return ( mDataPtr + ( mBitNum >> 3 ) ); }
	S32 getBytePos( ) const { return ( ( mBitNum + 7 ) >> 3 ); }
	S32 getBitPos( ) const { return ( mBitNum ); }
	bool getIsFull( ) const { return ( mBitNum >= mMaxBitNum ); }
	S32 getRemaining( ) const { return ( mBufSize - ( mBitNum >> 3 ) ); }
	S32 getSize( ) const { return ( mBufSize ); }
	bool getOk( ) { return ( !mError ); }
	
	// Read
	void read( void *buffer, int size ) { readBits( size << 3, buffer ); }
	void readAligned( void *buffer, int size );
	S32  readInt ( S32 bitCount );
	S32  readSignedInt ( S32 bitCount );	
	F32  readFloat( S32 bitCount );
	F32  readSignedFloat( S32 bitCount );
	void readNormalVector( S32 bitCount );
	void readBits( S32 bitCount, void *bitPtr );
	bool readFlag( );
	void readString( char stringBuf[ 256 ] );

	void setBit( S32 bitCount, bool set );

	// Write
	void write( void *buffer, int size ) { writeBits( size << 3, buffer ); }
	void writeAligned( void *buffer, int size );
	void writeBits( S32 bitCount, const void *bitPtr );
	void writeInt( S32 val, S32 bitCount );
	void writeFloat( F32 f, S32 bitCount );
	void writeSignedFloat( F32 f, S32 bitCount );
	void writeSignedInt( S32 value, S32 bitCount );
	bool writeFlag( bool val );
	void writeString( const char *string, S32 maxLen = 255 );
	
	void setBuffer( void *bufPtr, S32 bufSize );
	void setBitPos( const U32 bitPos ) { mBitNum = bitPos; }
};


#endif //__BITSTREAM_H__

