#ifndef __T1EVENTMANAGER_H__
#define __T1EVENTMANAGER_H__

/*
	- Crippled T1 Event Manager
	- Ignores most fields and can only pack SimConsoleEvents
*/

#include "T1.h"
#include "T1Game.h"
#include "bitStream.h"

#include <stdarg.h> // va_

// PlayerSayEvent
#define SAY_SYSTEM ( 0 )
#define SAY_GAME   ( 1 )
#define SAY_GLOBAL ( 2 )
#define SAY_TEAM   ( 3 )

#define REMOTESAY_GLOBAL ( 0 )
#define REMOTESAY_TEAM   ( 1 )

#define EVENT_SIMOBJECTTRANSFORM ( 0x405 )
#define EVENT_SIMCONSOLE         ( 0x408 )
#define EVENT_SIMPATH            ( 0x425 )
#define EVENT_PLAYERADD          ( 0x44b )
#define EVENT_PLAYERREMOVE       ( 0x44c )
#define EVENT_DELTASCORE         ( 0x44d )
#define EVENT_TEAMADD            ( 0x44f )
#define EVENT_MISSIONRESET       ( 0x450 )
#define EVENT_PLAYERTEAMCHANGE   ( 0x452 )
#define EVENT_PLAYERSAY          ( 0x453 )
#define EVENT_PLAYERCOMMAND      ( 0x455 )
#define EVENT_PLAYERSELECTCMDR   ( 0x456 )
#define EVENT_DATABLOCK          ( 0x458 )
#define EVENT_TEAMOBJECTIVE      ( 0x459 )
#define EVENT_LOCSOUND           ( 0x45c )
#define EVENT_TARGETNAME         ( 0x45d )
#define EVENT_VOICE              ( 0x45f )
#define EVENT_SOUND              ( 0x460 )
#define EVENT_PINGPL             ( 0x461 )
#define EVENT_PLAYERSKIN         ( 0x462 )

// DataBlocks
#define DB_SOUNDPROFILE   (  0 )
#define DB_SOUND          (  1 )
#define DB_DAMAGESKIN     (  2 )
#define DB_ARMOR          (  3 )
#define DB_STATICSHAPE    (  4 )
#define DB_ITEM           (  5 )
#define DB_ITEMIMAGE      (  6 )
#define DB_MOVEABLE       (  7 )
#define DB_SENSOR         (  8 )
#define DB_VEHICLE        (  9 )
#define DB_FLIER          ( 10 )
#define DB_TANK           ( 11 )
#define DB_HOVER          ( 12 )
#define DB_PROJECTILE     ( 13 )
#define DB_BULLET         ( 14 )
#define DB_GRENADE        ( 15 )
#define DB_ROCKET         ( 16 )
#define DB_LASER          ( 17 )
#define DB_INTERIORSHAPE  ( 18 )
#define DB_TURRET         ( 19 )
#define DB_EXPLOSION      ( 20 )
#define DB_MARKER         ( 21 )
#define DB_DEBRIS         ( 22 )
#define DB_MINE           ( 23 )
#define DB_TARGETLASER    ( 24 )
#define DB_SEEKINGMISSILE ( 25 )
#define DB_TRIGGER        ( 26 )
#define DB_CAR            ( 27 )
#define DB_LIGHTNING      ( 28 )
#define DB_REPAIREFFECT   ( 29 )
#define DB_IRCCHANNEL     ( 30 )

class T1Game;

/*
	SimConsoleEvent queue linked list

	Used for event queueing and packet notification. A better version would
	queue abstract Event objects each with their own id, pack, and unpack
	events, but that is overkill because we don't need to send anything
	other than SimConsoleEvents

	There are 32 queues because the send window for the low-level packet
	handler should freeze if there are 32 un-acked packets
*/

struct SimConsoleEvent {
	char *args[ 32 ];
	int argc;
	SimConsoleEvent *next;
};

class SimConsoleEventQueue {
public:
	SimConsoleEvent *mHead, *mTail;
	int mEventsQueued;
	bool mActive;

	void Empty( );
	void MergeFront( SimConsoleEventQueue &other_queue );
	void PushFront( SimConsoleEvent *e );
	void PushBack( SimConsoleEvent *e );
};



class T1EventManager {
public:
	T1EventManager( T1Game *game );
	~T1EventManager( );

	void ReadPacket( BitStream &packet );
	void WritePacket( BitStream &packet );
	void PacketNotify( bool recvd );

	void SimConsoleEvent_Queue( int argc, char *args[] );

private:
	T1Game *mGame;
	BitStream *mPacket;
	char mStringBuffer[ 257 ];
	
	SimConsoleEventQueue mSimConsoleEvents[ 32 ];
	int mEventIndex, mEventHead;
	
	bool ProcessEvent( int event_id );

	// DataBlocks
	void GameBaseData( );
	void ShapeBaseData(  );
	void StaticData(  );
	void StaticShapeData(  );
	void VehicleData(  );
	void ProjectileData(  );
	void SoundProfileData(  );
	void SoundData(  );
	void DamageSkinData(  );
	void ArmorData(  );
	void ItemData(  );
	void ItemImageData(  );
	void MoveableData(  );
	void SensorData(  );
	void FlierData(  );
	void BulletData(  );
	void GrenadeData(  );
	void RocketData(  );
	void LaserData(  );
	void TurretData(  );
	void ExplosionData(  );
	void MarkerData(  );
	void DebrisData(  );
	void SeekingMissileData(  );
	void LightningData(  );

	// Events	
	bool DataBlockEvent_Unpack(  );
	void DeltaScoreEvent_Unpack(  );
	void LocSoundEvent_Unpack(  );
	void MissionResetEvent_Unpack(  );
	void PingPLEvent_Unpack(  );
	void PlayerAddEvent_Unpack(  );
	void PlayerCommandEvent_Unpack(  );	
	void PlayerRemoveEvent_Unpack(  );
	void PlayerSayEvent_Unpack(  );
	void PlayerSelectCmdrEvent_Unpack( );
	void PlayerSkinEvent_Unpack(  );
	void PlayerTeamChangeEvent_Unpack(  );
	void SimConsoleEvent_Unpack(  );
	void SimConsoleEvent_Pack( SimConsoleEvent &e );	
	void SimEvent_Unpack(  );
	void SimObjectTransformEvent_Unpack( );
	void SimPathEvent_Unpack(  );
	void SoundEvent_Unpack(  );
	void TargetNameEvent_Unpack(  );
	void TeamAddEvent_Unpack(  );
	void TeamObjectiveEvent_Unpack(  );	
	void VoiceEvent_Unpack(  );	
};




#endif // __T1EVENTMANAGER_H__
