#include "T1NetStream.h"
#include "Console.h"

T1NetStream::T1NetStream( T1Game *game ) {
	mGame = game;
	mEventManager = new T1EventManager( game );
	mPacketRate = mMaxPacketRate = ( 1000 / 16 );
	mPacketSize = mMaxPacketSize = ( 400 );
	mPacketFrame = ( 1000 / 31 );
	mRateChanged = mMaxRateChanged = true;
	mStartTicks = T1::GetSystemTicks( );
	mLastUpdate = 0;
	
	SetDemoName( "" );
}

T1NetStream::~T1NetStream( ) {
	delete mEventManager;
}

bool T1NetStream::GetOk( ) {
	return ( mOk );
}

int T1NetStream::GetTicks( ) {
	return ( T1::GetSystemTicks( ) - mStartTicks );
}

void T1NetStream::OnConnectionAccepted( BitStream &stream ) {
	int manager_id = stream.readInt( 32 );
	int server_id = stream.readInt( 32 );

	mGame->OnConnectionAccepted( manager_id, server_id );
}

void T1NetStream::OnConnectionRejected( const char *reason ) {
	mGame->OnConnectionRejected( reason );
}

void T1NetStream::OnConnectionDisconnected( const char *reason ) {
	mGame->OnDisconnected( reason );
}

void T1NetStream::OnConnectionTimedOut( ) {
	mGame->OnConnectionTimedOut( );
}

void T1NetStream::OnReadPacket( BitStream &stream ) {
}

// Alert the NetStream that a packet was either received or dropped
void T1NetStream::OnPacketNotify( bool recvd ) {
	mEventManager->PacketNotify( recvd );
}

void T1NetStream::ReadPacket( BitStream &packet ) {
	OnReadPacket( packet );

	if ( packet.readFlag( ) ) {
		mPacketRate = packet.readInt( 10 );
		mPacketSize = packet.readInt( 10 );
	}

	if ( packet.readFlag( ) ) {
		int rate = packet.readInt( 10 );
		int size = packet.readInt( 10 );

		if ( rate < mMaxPacketRate )
			rate = mMaxPacketRate;
		if ( size > mMaxPacketSize )
			size = mMaxPacketSize;

		if ( rate != mPacketRate || size != mPacketSize ) {
			mPacketRate = rate;
			mPacketSize = size;
			mRateChanged = true;
		}
	}

	if ( packet.readFlag( ) )
		mEventManager->ReadPacket( packet );

	if ( !packet.getOk( ) ) {
		// We could quit here, but we'll just hope the packets get better
		Console::Error( "BitStream overflow in T1NetStream::ReadPacket\n" );
	}
}

// Does nothing by default, override if you want to send remotes
void T1NetStream::RemoteEval( char *function, int argc, ... ) {
}

void T1NetStream::SetDemoName( char *name ) {
	T1::Strncpy( mDemoName, name, 254 );
}

bool T1NetStream::WritePacket( BitStream &packet ) {
	int now = T1::GetSystemTicks( );
	if ( now - mLastUpdate < mPacketFrame )
		return ( false );
	mLastUpdate = now;

	if ( packet.writeFlag( mRateChanged ) ) {
		packet.writeInt( mPacketRate, 10 );
		packet.writeInt( mPacketSize, 10 );
		mRateChanged = false;
	}

	if ( packet.writeFlag( mMaxRateChanged ) ) {
		packet.writeInt( mMaxPacketRate, 10 );
		packet.writeInt( mMaxPacketSize, 10 );
		mMaxRateChanged = false;
	}

	mEventManager->WritePacket( packet );
	
	packet.writeFlag( false ); // playerpsc
	packet.writeFlag( false ); // ghostmanager

	return ( true );
}
