#include "T1.h"
#include "T1Game.h"
#include "Console.h"

// Example game class
class mygame : public T1Game {
private:
	class team {
	public:
		int id;
		char name[ 32 ];
	};

	class player {
	public:
		int id;
		char name[ 32 ];
		int team;
		int ping;
		float pl;
	};

public:
	mygame::mygame( ) : T1Game( ) {
		memset( mTeams, 0, sizeof( mTeams ) );
		memset( mPlayers, 0, sizeof( mPlayers ) );

		// Dummy player for the server
		mServer.id = 2048;
		T1::Strncpy( mServer.name, "", 31 );
	}

	bool mygame::Start( ) {
		mTicksMapStart = GetGameTicks( );
		bool ret = T1Game::Start( );		

		return ( ret );
	}

	/*
		Utility functions
	*/

	player *getplayer( int id ) {
		if ( id == 0 )
			return ( &mServer );
		else  if ( id <= 2048 || id >= ( 2049 + 128 ) )
			return ( NULL );

		return ( &mPlayers[ id - 2048 ] );
	}

	team *getteam( int id ) {
		if ( id < -1 || id >= 32 )
			return ( NULL );
		return ( &mTeams[ id + 1 ] );
	}

	char *getchattype( int type ) {
		switch ( type ) {
			case SAY_SYSTEM:
			case SAY_GAME:   return ( "Server" );
			case SAY_GLOBAL: return ( "Global" );
			case SAY_TEAM:   return ( "Teamchat" );
		}

		return ( "Unpossible!" );
	}

	char *getdatablock( int type ) {
		switch ( type ) {
			case DB_SOUNDPROFILE: return ( "SoundProfile" );
			case DB_SOUND: return ( "Sound" );
			case DB_DAMAGESKIN: return ( "DamageSkin" );
			case DB_ARMOR: return ( "Armor" );
			case DB_STATICSHAPE: return ( "StaticShape" );
			case DB_ITEM: return ( "Item" );
			case DB_ITEMIMAGE: return ( "ItemImage" );
			case DB_MOVEABLE: return ( "Moveable" );
			case DB_SENSOR: return ( "Sensor" );
			case DB_VEHICLE: return ( "Vehicle" );
			case DB_FLIER: return ( "Flier" );
			case DB_TANK: return ( "Tank" );
			case DB_HOVER: return ( "Hover" );
			case DB_PROJECTILE: return ( "Projectile" );
			case DB_BULLET: return ( "Bullet" );
			case DB_GRENADE: return ( "Grenade" );
			case DB_ROCKET: return ( "Rocket" );
			case DB_LASER: return ( "Laser" );
			case DB_INTERIORSHAPE: return ( "InteriorShape" );
			case DB_TURRET: return ( "Turret" );
			case DB_EXPLOSION: return ( "Explosion" );
			case DB_MARKER: return ( "Marker" );
			case DB_DEBRIS: return ( "Debris" );
			case DB_MINE: return ( "Mine" );
			case DB_TARGETLASER: return ( "TargetLaser" );
			case DB_SEEKINGMISSILE: return ( "SeekingMissile" );
			case DB_TRIGGER: return ( "Trigger" );
			case DB_CAR: return ( "Car" );
			case DB_LIGHTNING: return ( "Lightning" );
			case DB_REPAIREFFECT: return ( "RepairEffect" );
			case DB_IRCCHANNEL: return ( "IRCChannel" );
		}
		
		return ( "Unpossible!" );
	}

	void log_timestamp( ) {
		int ticks = ( GetGameTicks( ) - mTicksMapStart );
		int hours, mins, secs, ms;
		hours = ( ticks           ) / 3600000;
		mins  = ( ticks % 3600000 ) /   60000;
		secs  = ( ticks %   60000 ) /    1000;
		ms    = ( ticks %    1000 ) /      10;

		Console::Print( "[%02d:%02d:%02d.%02d] ", hours, mins, secs, ms );
	}

	void log( char *fmt, ... ) {
		log_timestamp( );		

		va_list args;
		va_start( args, fmt );
		Console::PrintVarArg( fmt, args );
		Console::Print( "\n" );
		va_end( args );
	}

	void log_color( int color, char *fmt, ... ) {
		log_timestamp( );

		va_list args;
		va_start( args, fmt );
		Console::PrintColorVarArg( color, fmt, args );
		Console::Print( "\n" );
		va_end( args );
	}

	// Switch the bot to observer
	void change_to_observer( ) {
		mNetStream->RemoteEval( "scoresOn", 0 );
		mNetStream->RemoteEval( "menuSelect", 1, "changeteams" );
		mNetStream->RemoteEval( "menuSelect", 1, "-2" );
	}

	// equivalent of say( type, msg ); on a Tribes client
	void say( int type, char *fmt, ... ) {
		char msg_text[ 2048 ];
		char msg_type[ 8 ];
		
		snprintf( msg_type, 8, "%d", type );
		
		va_list args;
		va_start( args, fmt );
		vsnprintf( msg_text, 2048, fmt, args );
		va_end( args );

		mNetStream->RemoteEval( "say", 2, msg_type, msg_text );
	}

	/*
		Game Event Callbacks
	*/
	virtual void OnConnectionAccepted( int manager_id, int server_id ) {
		mManagerId = manager_id;
		mServerId = server_id;
		mConnected = true;

		log( "CONNECTION: accepted, manager = %d, server = %d", mManagerId, mServerId );
	}

	virtual void OnConnectionRejected( const char *reason  ) {
		mConnected = false;
		mRetryTicks = GetGameTicks( );
		log( "CONNECTION: rejected, server said %s", reason );				
	}

	virtual void OnConnectionTimedOut( ) {
		mConnected = false;
		mRetryTicks = GetGameTicks( );
		log( "CONNECTION: timed out" );
	}

	virtual void OnDisconnected( const char *reason ) {
		mConnected = false;
		mRetryTicks = GetGameTicks( );
		log( "CONNECTION: disconnected, server said %s", reason );
	}

	virtual void OnDataBlock( int type, int index, int index_max ) {
		char *name = getdatablock( type );
		if ( index == 0xff )
			log( "DATABLOCK: %s (no items)", name );
		else
			log( "DATABLOCK: %s %d of %d", name, index+1, index_max );
	}

	// Overriding OnDataFinished must chain to T1Game::OnDataFinished
	virtual void OnDataFinished( ) {
		T1Game::OnDataFinished( );

		log( "DATABLOCK: Data Finished" );
	}

	virtual void OnPacket( int size ) {
		// log( "Packet, %d bytes, %d bytes in", size, bytes_in );
	}

	virtual void OnPlayerPingPL( int id_player, int ping, float pl ) {
		player *p = getplayer( id_player );
		if ( !p )
			return;

		p->ping = ping;
		p->pl = pl;
	}

	virtual void OnPlayerChangeTeam( int id_player, int id_team ) {
		player *p = getplayer( id_player );
		team *t = getteam( id_team );
		if ( !p || !t )
			return;

		team *old_t = getteam( p->team );
		if ( !old_t )
			return;

		// Change the bot to observer 
		if ( ( p->id == mManagerId )  && ( id_team != -1 ) )
			change_to_observer( );


		p->team = id_team;
		if ( id_team == -1 )
			log( "TEAMCHANGE: %s is no longer on a team", p->name );
		else
			log( "TEAMCHANGE: %s changed from %s to %s", p->name, old_t->name, t->name );
	}


	virtual void OnPlayerDrop( int id_player ) {
		player *p = getplayer( id_player );
		if ( !p || !p->id )
			return;

		p->id = 0;
		log( "DROP: %s dropped", p->name );
	}

	virtual void OnPlayerJoin( int id_player, int id_team, char *name ) {
		player *p = getplayer( id_player );
		team *t = getteam( id_team );
		if ( !p || !t )
			return;

		p->id = id_player;
		p->team = id_team;
		p->ping = 0;
		p->pl = 0.0f;
		T1::Strncpy( p->name, name, 31 );

		log( "JOIN: Player %d(%s) joined on team %s", p->id, p->name, t->name );
	}

	virtual void OnPlayerSay( int id_player, int type, char *msg ) {
		player *p = getplayer( id_player );
		if ( !p || !p->id )
			return;

		if ( id_player == 0 ) {
			if ( !strcmp( msg, "Match started." ) )
				mTicksMapStart = GetGameTicks( );
		} else {
			if ( !strcmp( msg, "!test" ) )
				say( REMOTESAY_GLOBAL, "Bot compiled on %s for %s", __DATE__, __OS__ );
		}

		int msg_color = CONSOLE_COLOR_DEFAULT;

		switch ( type ) {
			case SAY_SYSTEM: msg_color = CONSOLE_WHITE; break;
			case SAY_GAME: msg_color = CONSOLE_RED; break;
			case SAY_GLOBAL: msg_color = CONSOLE_YELLOW; break;
			case SAY_TEAM: msg_color = CONSOLE_GREEN; break;
		}

		log_color( msg_color, "SAY: %s (%s): %s", p->name, getchattype( type ), msg );
	}

	// Overriding OnMissionReset must chain to T1Game::OnMissionReset
	virtual void OnMissionReset( ) {
		T1Game::OnMissionReset( );

		log( "MISSIONRESET" );
		mTicksMapStart = GetGameTicks( );
	}

	virtual void OnRemoteEval( char *function, int argc, char *argv[] ) {
		log_color( CONSOLE_MAGENTA, "REMOTEEVAL: %s (%d args)", function, argc );
	}

	// Overriding OnTeamAdd must chain to T1Game::OnTeamAdd
	virtual void OnTeamAdd( int id_team, char *name ) {
		T1Game::OnTeamAdd( id_team, name );

		team *t = getteam( id_team );
		if ( !t )
			return;

		t->id = id_team;
		if ( t->id == -1 ) 
			T1::Strncpy( t->name, "Observer", 31 );
		else
			T1::Strncpy( t->name, name, 31 );
		log( "TEAMADD: %s (id %d)", t->name, t->id );
	}


	// Overriding Poll must chain to T1Game::Poll
	virtual bool Poll( ) {		
		bool ret = T1Game::Poll( );

		if ( !GetIsDemo( ) ) {
			int now = GetGameTicks( );
			if ( !mConnected && ( now - mRetryTicks > 3000 ) ) {
				mRetryTicks = now;

				log_color( CONSOLE_GREEN, "Retrying connection to %s..", mLastServer );
				OpenLastServer( );
				Start( ); // restart connection
			}

			// Poll never returns false if we aren't in a demo, user must exit manually
			ret = true;
		}

		return ( ret );
	}

private:
	team mTeams[ 32 ];
	player mServer, mPlayers[ 128 ];
	int mTicksMapStart, mRetryTicks;
};


#if !defined(__WIN32__)
	#include <signal.h>
#endif

// Signal handler to allow us to quit with various methods
class SignalHandler {
public:
	SignalHandler( ) {
		need_to_quit = false;
		install_signals( );		
	}

	static bool kbhit( );
	static void install_signals( );
	static void signal_handler( int signal_number );

private:
	static bool need_to_quit;
};

bool SignalHandler::need_to_quit;

bool SignalHandler::kbhit( ) {
#if defined(__WIN32__)
	need_to_quit = ( _kbhit( ) != 0 );
#endif

	return ( need_to_quit );
}

void SignalHandler::signal_handler( int signal_number ) {
#if !defined(__WIN32__)
	signal( signal_number, SIG_DFL );
	SignalHandler::need_to_quit = true;
#endif
}

void SignalHandler::install_signals( ) {
#if !defined(__WIN32__)
	// ctrl-c, hup, etc
	signal( SIGHUP, SignalHandler::signal_handler );
	signal( SIGINT, SignalHandler::signal_handler );
	signal( SIGQUIT, SignalHandler::signal_handler );
	signal( SIGUSR1, SignalHandler::signal_handler );
	signal( SIGTERM, SignalHandler::signal_handler );
#endif
}

int main( int argc, char *argv[] ) {
	SignalHandler signals;
	Console::Init( );
	T1::Net_Startup( );	

	mygame *game = new mygame( );

	// game->OpenServer( "IP:127.0.0.1:28001", "demo.rec" );
	// game->OpenServer( "IP:127.0.0.1:28001" );

	game->OpenDemo( "demo.rec" );

	// Only applies to demos
	game->SetFullspeed( true );

	if ( game->Start( ) ) {
		while ( game->Poll( ) && !signals.kbhit( ) ) {
			T1::Sleep( 25 );
		}
	}

	delete game;	
	T1::Net_Shutdown( );
	Console::Close( );

	return ( 0 );
}
