#include "T1EventManager.h"
#include "Console.h"

// Empty the queue
void SimConsoleEventQueue::Empty( ) {
	while ( mHead ) {
		SimConsoleEvent *next = mHead->next;
		free( mHead );
		mHead = next;
	}

	mHead = mTail = NULL;
	mEventsQueued = 0;
}

void SimConsoleEventQueue::MergeFront( SimConsoleEventQueue &other_queue ) {
	if ( other_queue.mEventsQueued <= 0 )
		return;

	if ( !mHead )
		mTail = other_queue.mTail;
	else
		other_queue.mTail->next = mHead;

	mHead = other_queue.mHead;		
	mEventsQueued += other_queue.mEventsQueued;
}

void SimConsoleEventQueue::PushFront( SimConsoleEvent *e ) {
	if ( !mHead )
		mTail = e;
	else
		e->next = mHead;

	mHead = e;
	mEventsQueued++;
}

void SimConsoleEventQueue::PushBack( SimConsoleEvent *e ) {
	if ( !mHead )
		mHead = e;
	else
		mTail->next = e;

	mTail = e;
	mEventsQueued++;
}


/*
	T1EventManager
*/

T1EventManager::T1EventManager( T1Game *game ) {
	mGame = game;
	memset( mSimConsoleEvents, 0, sizeof( mSimConsoleEvents ) );

	/*
		mEventIndex is the current event queue to write to if we
		want to send an event

		mEventHead is the packet notification index so we can
		resend any dropped events.

		If the server doesn't respond until we've written 5
		packets, mEventIndex will be at 5, but mEventHead
		will still be at 0 when we get a drop/recv notification
	*/

	mEventIndex = 0;
	mEventHead = 0;
}

T1EventManager::~T1EventManager( ) {
	// free all queues
	for ( int i = 0; i < 32; i++ )
		mSimConsoleEvents[ i ].Empty( );
}

void T1EventManager::ReadPacket( BitStream &packet ) {
	mPacket = &packet;

	while ( mPacket->readFlag( ) ) {
		// sequencing, we'll attempt to ignore it and just plow through
		if ( mPacket->readFlag( ) ) {
			if ( !mPacket->readFlag( ) ) {
				if ( mPacket->readFlag( ) )
					mPacket->readInt( 7 );
			}
		}

		int event_id = ( mPacket->readInt( 7 ) + 0x400 );
		if ( !ProcessEvent( event_id ) )
			return;
	}
}

void T1EventManager::WritePacket( BitStream &packet ) {
	mPacket = &packet;	

	SimConsoleEventQueue &q = mSimConsoleEvents[ mEventIndex ];
	
	if ( q.mActive ) {
		Console::Error( "EventManager: Event queue wrapped, why didn't IsWindowFull stop this?\n" );
		mPacket->writeFlag( false );
		return;
	}

	q.mActive = true;
	SimConsoleEvent *walk = q.mHead;	
	mEventIndex = ( mEventIndex + 1 ) & 31;

	// are there any events to write out?
	if ( walk ) {
		mPacket->writeFlag( true );

		for ( int events = 0; walk; walk = walk->next, events++ ) {
			// ignore ordering
			mPacket->writeFlag( true );
			mPacket->writeFlag( false );

			mPacket->writeInt( EVENT_SIMCONSOLE - 0x400, 7 );
			SimConsoleEvent_Pack( *walk );

			// We normally shouldn't go this overboard
			// Peg the end of the queue onto the front of the next queue
			if ( mPacket->getBytePos( ) > 800 ) {	
				SimConsoleEventQueue &q_next = mSimConsoleEvents[ mEventIndex ];
				q_next.mEventsQueued += ( q.mEventsQueued - events );
				q.mEventsQueued = events;
				q_next.PushFront( walk->next ); // splice the remaining in
				walk->next = NULL;
			}
		}
	}
	
	mPacket->writeFlag( false );
}

// PacketNotify is called every time a packet drop/recv event is received
void T1EventManager::PacketNotify( bool recvd ) {
	SimConsoleEventQueue &q = mSimConsoleEvents[ mEventHead ];
	mEventHead = ( mEventHead + 1 ) & 31;

	if ( recvd ) {
		// recvd, remove event from head queue
		q.Empty( );	
	} else {
		// add dropped events to current queue
		mSimConsoleEvents[ mEventIndex ].MergeFront( q ); 
		
		// don't free old queue, events still exist in current
		q.mEventsQueued = 0;
		q.mHead = q.mTail = NULL;
	}

	q.mActive = false;
}


// Add a SimConsoleEvent (remoteEval) to the Event queue for the next packet write
void T1EventManager::SimConsoleEvent_Queue( int argc, char *args[] ) {
	size_t size = ( 0 );
	int i;

	// sum up size of args
	for ( i = 0; i < argc; i++ )
		size += ( strlen( args[ i ] ) + 1 );

	char *data = (char *)malloc( sizeof( SimConsoleEvent ) + size );	
	if ( !data )
		return;
	SimConsoleEvent *e = (SimConsoleEvent *)data;
	data += sizeof( SimConsoleEvent );
	e->next = NULL;
	e->argc = argc;

	for ( i = 0; i < argc; i++ ) {
		size_t len = strlen( args[ i ] );
		T1::Strncpy( data, args[ i ], len );
		e->args[ i ] = data;
		data += ( len + 1 );
	}

	mSimConsoleEvents[ mEventIndex ].PushBack( e );
}


bool T1EventManager::ProcessEvent( int event_id ) {
	switch ( event_id ) {
		case 0x405: SimObjectTransformEvent_Unpack( ); break;
		case 0x408: SimConsoleEvent_Unpack( ); break;
		case 0x425: SimPathEvent_Unpack( ); break;	
		case 0x44b: PlayerAddEvent_Unpack( ); break;
		case 0x44c: PlayerRemoveEvent_Unpack( ); break;
		case 0x44d: DeltaScoreEvent_Unpack( ); break;
		case 0x44f: TeamAddEvent_Unpack( ); break;
		case 0x450: MissionResetEvent_Unpack( ); break;
		case 0x452: PlayerTeamChangeEvent_Unpack( ); break;
		case 0x453: PlayerSayEvent_Unpack( ); break;
		case 0x455: PlayerCommandEvent_Unpack( ); break;
		case 0x456: PlayerSelectCmdrEvent_Unpack( ); break;
		case 0x458: return( DataBlockEvent_Unpack( ) ); break;
		case 0x459: TeamObjectiveEvent_Unpack( ); break;
		case 0x45c: LocSoundEvent_Unpack( ); break;
		case 0x45d: TargetNameEvent_Unpack( ); break;
		case 0x45f: VoiceEvent_Unpack( ); break;
		case 0x460: SoundEvent_Unpack( ); break;
		case 0x461: PingPLEvent_Unpack( ); break;
		case 0x462: PlayerSkinEvent_Unpack( ); break;
	
		// we couldn't handle it, bail
		default: return ( false );
	}

	return ( true );
}


/*
	DataBlocks from the server. Most fields are ignored and just
	bulk read since they are of little use. The only time we can't
	do this is when there's a string to be read

	Inherited blocks first, other DataBlocks build off these
*/

void T1EventManager::GameBaseData( ) {
	mPacket->readSignedInt( 8 );
	mPacket->readString( mStringBuffer ); // map icon
	mPacket->readString( mStringBuffer ); // class name
}

void T1EventManager::ShapeBaseData( ) {
	mPacket->readString( mStringBuffer ); // shape
	if ( mStringBuffer[ 0 ] == '\xfe' || mStringBuffer[ 0 ] == '\xff' )
		mPacket->burnBits( 32 );
}

void T1EventManager::StaticData( ) {
	GameBaseData( );
	ShapeBaseData( );

	mPacket->readString( mStringBuffer ); // shield name
	mPacket->burnBits( 96 );

	for ( int i = 0; i < 16; i++ ) {
		if ( mPacket->readFlag( ) ) {
			mPacket->readString( mStringBuffer ); // action
			mPacket->readInt( 8 );
		}
	}

	mPacket->readFlag( );
}

void T1EventManager::StaticShapeData( ) {
	StaticData( );
	
	mPacket->readInt( 8 );	
	mPacket->readFlag( );
	mPacket->readFlag( );
}

void T1EventManager::VehicleData( ) {
	StaticShapeData( );

	mPacket->burnBits( 632 );
}

void T1EventManager::ProjectileData( ) {
	GameBaseData( );
	ShapeBaseData( );

	mPacket->burnBits( 546 );
}

/*
	Child blocks, nobody inherits off these
*/

void T1EventManager::SoundProfileData( ) {
	mPacket->readInt( 6 );
	mPacket->readFloat( 10 );

	if ( mPacket->readFlag( ) )
		mPacket->readFloat( 10 );
	if ( mPacket->readFlag( ) )
		mPacket->readFloat( 10 );
	if ( mPacket->readFlag( ) )
		mPacket->readNormalVector( 10 );
	if ( mPacket->readFlag( ) )
		mPacket->burnBits( 32 );
	if ( mPacket->readFlag( ) )
		mPacket->burnBits( 32 );
	if ( mPacket->readFlag( ) )
		mPacket->burnBits( 32 );
}

void T1EventManager::SoundData( ) {
	mPacket->readString( mStringBuffer ); // name
	mPacket->readFloat( 6 );
	mPacket->readInt( 8 );
}

void T1EventManager::DamageSkinData( ) {
	for ( int i = 0; i < 10; i++ )
		mPacket->readString( mStringBuffer ); // skin[ i ]
}

// ArmorData blocks use a ton of bandwidth
void T1EventManager::ArmorData( ) {
	StaticData( );

	mPacket->readString( mStringBuffer ); // jet_flame

	for ( int i = 0; i < 51; i++ ) {
		mPacket->readString( mStringBuffer ); // anim_node
		if ( mPacket->readFlag( ) )
			mPacket->readInt( 8 );

		mPacket->readFlag( );
		mPacket->readInt( 4 );
		mPacket->readInt( 4 );
	}

	mPacket->burnBits( 807 );
}

// Items use a bit of bandwidth too
void T1EventManager::ItemData( ) {
	StaticData( );

	mPacket->burnBits( 224 );
	mPacket->readString( mStringBuffer );
	mPacket->readFlag( );
	mPacket->readFlag( );
	mPacket->readFlag( );
	mPacket->readString( mStringBuffer );
	mPacket->burnBits( 192 );
}

// Another bandwidth sucker
void T1EventManager::ItemImageData( ) {
	GameBaseData( );

	mPacket->readString( mStringBuffer ); // item name
	mPacket->burnBits( 849 );
}

void T1EventManager::MoveableData( ) {
	StaticShapeData( );
	
	mPacket->burnBits( 67 );
}

void T1EventManager::SensorData( ) {
	StaticShapeData( );
	
	mPacket->burnBits( 66 );
}

void T1EventManager::FlierData( ) {
	VehicleData( );
	
	mPacket->burnBits( 224 );
}

void T1EventManager::BulletData( ) {
	ProjectileData( );

	mPacket->burnBits( 96 );
}

void T1EventManager::GrenadeData( ) {
	ProjectileData( );

	mPacket->burnBits( 32 );
	mPacket->readString( mStringBuffer ); // trail
}

void T1EventManager::RocketData( ) {
	ProjectileData( );

	mPacket->burnBits( 32 );
	mPacket->readString( mStringBuffer ); // trail
	mPacket->burnBits( 64 );
}

void T1EventManager::LaserData( ) {
	ProjectileData( );

	mPacket->readString( mStringBuffer ); // pulse
	mPacket->readString( mStringBuffer ); // hit model
	mPacket->readFlag( );
}

void T1EventManager::TurretData( ) {
	SensorData( );

	mPacket->burnBits( 225 );
}

void T1EventManager::ExplosionData( ) {
	GameBaseData( );
	ShapeBaseData( );

	mPacket->burnBits( 66 );
	if ( mPacket->readFlag( ) )
		mPacket->burnBits( 480 );
	mPacket->readFlag( );
}

void T1EventManager::MarkerData( ) {
	GameBaseData( );

	mPacket->readString( mStringBuffer ); // icon
}

void T1EventManager::DebrisData( ) {
	GameBaseData( );

	mPacket->burnBits( 32 );
	mPacket->readString( mStringBuffer );
	mPacket->burnBits( 707 );
}

void T1EventManager::SeekingMissileData( ) {
	ProjectileData( );

	mPacket->burnBits( 96 );
}

void T1EventManager::LightningData( ) {
	ProjectileData( );

	mPacket->burnBits( 192 );
	mPacket->readString( mStringBuffer ); // bmp
}

/*
	Game Events are triggered by the demopacket to cause non-world interactions
	such as messages, remoteEvals, player joins/drops, etc.
*/

bool T1EventManager::DataBlockEvent_Unpack( ) {
	int db_type = mPacket->readInt( 6 );
	int db_maxindex = mPacket->readInt( 8 );
	int db_index = mPacket->readInt( 8 );

	mGame->OnDataBlock( db_type, db_index, db_maxindex );

	if ( db_index != 0xff ) {
		switch ( db_type ) {
			case DB_SOUNDPROFILE: SoundProfileData( ); break;
			case DB_SOUND: SoundData( ); break;
			case DB_DAMAGESKIN: DamageSkinData( ); break;
			case DB_ARMOR: ArmorData( ); break;
			case DB_STATICSHAPE: StaticShapeData( ); break;
			case DB_ITEM: ItemData( ); break;
			case DB_ITEMIMAGE: ItemImageData( ); break;
			case DB_MOVEABLE: MoveableData( ); break;
			case DB_SENSOR: SensorData( ); break;
			case DB_FLIER: FlierData( ); break;
			case DB_BULLET: BulletData( ); break;
			case DB_GRENADE: GrenadeData( ); break;
			case DB_ROCKET: RocketData( ); break;
			case DB_LASER: LaserData( ); break;
			case DB_TURRET: TurretData( ); break;
			case DB_EXPLOSION: ExplosionData( ); break;
			case DB_MARKER: MarkerData( ); break;
			case DB_DEBRIS: DebrisData( ); break;
			case DB_MINE: ItemData( ); break;
			case DB_TARGETLASER: LaserData( ); break;
			case DB_SEEKINGMISSILE: SeekingMissileData( ); break;
			case DB_TRIGGER: GameBaseData( ); break;
			case DB_LIGHTNING: LightningData( ); break;
			case DB_REPAIREFFECT: LightningData( ); break;
			case DB_IRCCHANNEL: GameBaseData( ); break;

			default: return ( false );
		}
	}

	if ( db_type == DB_IRCCHANNEL ) {
		if ( ( db_index == 0xff ) || ( db_index == ( db_maxindex - 1 ) ) ) {
			mGame->OnDataFinished( );
		}
	}

	return ( true );
}

void T1EventManager::DeltaScoreEvent_Unpack( ) {
	if ( mPacket->readFlag( ) )
		mPacket->readInt( 4 ); // team
	else
		mPacket->readInt( 7 ); // player id + 0x800
	mPacket->readInt( 32 ); // score
	mPacket->readString( mStringBuffer ); // status line
}

void T1EventManager::LocSoundEvent_Unpack( ) {
	mPacket->readInt( 8 ); // id

	if ( mPacket->readFlag( ) ) {
		mPacket->readNormalVector( 8 );
		mPacket->readFloat( 9 );
	}
}

void T1EventManager::MissionResetEvent_Unpack( ) {
	mGame->OnMissionReset( );
}


void T1EventManager::PingPLEvent_Unpack( ) {
	int items = mPacket->readInt( 7 );
	
	for ( int i = 0; i < items; i++ ) {
		int id = mPacket->readInt( 7 ) + 0x800;
		int ping = mPacket->readInt( 8 ) << 2;
		float pl = mPacket->readFloat( 6 );

		mGame->OnPlayerPingPL( id, ping, pl );
	}
}


void T1EventManager::PlayerAddEvent_Unpack( ) {
	int id_player, id_team;
	char name[ 257 ], skin[ 257 ], voice[ 257 ];

	id_player = mPacket->readInt( 32 );
	mPacket->readFlag( );
	mPacket->readString( name );
	mPacket->readString( skin );
	mPacket->readString( voice );
	mPacket->readInt( 1 );
	mPacket->burnBits( 32 );
	if ( mPacket->readFlag( ) )
		id_team = mPacket->readInt( 3 );
	else
		id_team = -1;

	mGame->OnPlayerJoin( id_player, id_team, name );
}

void T1EventManager::PlayerCommandEvent_Unpack( ) {
	mPacket->readInt( 10 ); // player_id = readInt( 10 ) + 0x800
	mPacket->readInt( 2 ); // type
	int giver_id = -1;

	if ( !mPacket->readFlag( ) && mPacket->readFlag( ) ) {
		giver_id = mPacket->readInt( 10 ) + 0x800;
		mPacket->readInt( 6 );

		if ( mPacket->readFlag( ) )
			mPacket->readInt( 8 );

		mPacket->readInt( 10 );
		mPacket->readInt( 10 );
	}

	mStringBuffer[ 0 ] = '\x0';
	if ( mPacket->readFlag( ) && mPacket->readFlag( ) ) {
		mPacket->readString( mStringBuffer );
	}
}


void T1EventManager::PlayerRemoveEvent_Unpack( ) {
	mGame->OnPlayerDrop( mPacket->readInt( 7 ) + 0x800 );
}


void T1EventManager::PlayerSayEvent_Unpack( ) {
	int id = mPacket->readInt( 12 );
	int type = mPacket->readInt( 5 );
	mPacket->readString( mStringBuffer );

	if ( id == 2048 )
		id = 0;

	mGame->OnPlayerSay( id, type, mStringBuffer );
}

void T1EventManager::PlayerSelectCmdrEvent_Unpack( ) {
	mPacket->readInt( 7 ); // + 0x800
	if ( mPacket->readFlag( ) )
		mPacket->readInt( 7 ); // + 0x800
}

void T1EventManager::PlayerSkinEvent_Unpack( ) {
	mPacket->readInt( 7 ); // player_id = readInt( 7 ) + 0x800
	mPacket->readString( mStringBuffer ); // skin?
}

void T1EventManager::PlayerTeamChangeEvent_Unpack( ) {
	int id = mPacket->readInt( 7 ) + 0x800;
	int team = mPacket->readInt( 4 ) - 1;

	mGame->OnPlayerChangeTeam( id, team );
}


void T1EventManager::SimConsoleEvent_Pack( SimConsoleEvent &e ) {
	mPacket->writeInt( e.argc, 5 );

	for ( int i = 0; i < e.argc; i++ )
		mPacket->writeString( e.args[ i ] );
}

void T1EventManager::SimConsoleEvent_Unpack( ) {
	int argc = ( mPacket->readInt( 5 ) - 1 ); // -1 from the function name
	char cmd[ 257 ];
	char *argv[ 255 ];
	char args[ 2048 ];
	size_t str_pos = 0;

	if ( argc == -1 ) // can it really have nothing?
		return;

	mPacket->readString( cmd );

	for ( int i = 0; i < argc; i++ ) {		
		mPacket->readString( &args[ str_pos ] );

		argv[ i ] = &args[ str_pos ];
		str_pos += strlen( &args[ str_pos ] ) + 1;
	}

	mGame->OnRemoteEval( cmd, argc, argv );
}

void T1EventManager::SimEvent_Unpack( ) {
	if ( mPacket->readFlag( ) )
		mPacket->readInt( 10 );
	else if ( mPacket->readFlag( ) )
		mPacket->readInt( 10 );
	else
		mPacket->readInt( 32 );
}

void T1EventManager::SimObjectTransformEvent_Unpack( ) {
	SimEvent_Unpack( );
}

void T1EventManager::SimPathEvent_Unpack( ) {
	SimEvent_Unpack( );

	mPacket->burnBits( 32 );
	if ( mPacket->readFlag( ) ) {
		mPacket->readFlag( );
		int count = mPacket->readInt( 32 );

		for ( int i = 0; i < count; i++ )
			mPacket->burnBits( 384 );
	}
}

void T1EventManager::SoundEvent_Unpack( ) {
	mPacket->readInt( 10 );
	mPacket->readInt( 2 );	
	mPacket->readInt( 8 );
}

void T1EventManager::TargetNameEvent_Unpack( ) {
	mPacket->readInt( 7 ); // index
	mPacket->readString( mStringBuffer ); // target name
}


void T1EventManager::TeamAddEvent_Unpack( ) {
	int id_team;
	char name[ 257 ], skin[ 257 ];
	
	mPacket->burnBits( 32 );
	id_team = mPacket->readInt( 32 );
	mPacket->readString( name );
	mPacket->readString( skin );

	mGame->OnTeamAdd( id_team, name );
}

void T1EventManager::TeamObjectiveEvent_Unpack( ) {
	mPacket->readInt( 32 ); // line on objective screen
	mPacket->readString( mStringBuffer ); // line text
}

void T1EventManager::VoiceEvent_Unpack( ) {
	mPacket->readInt( 7 ); // player_id = readInt( 7 ) + 0x800
	mPacket->readString( mStringBuffer ); // wav
}
