/*
 
   $Workfile:   bitStream.cpp  $
 
   DESCRIPTION:
 
 
   (c) Copyright 1996, Dynamix Inc.   All rights reserved.
 
  */

// You really don't want to remove this
#define __BITSTREAM_RANGE_CHECK__

#include "bitStream.h"
#include "Console.h"

// Not the best way but nobody is looking
HuffmanProcessor HuffmanProcessor::gHuffProcessor;

void BitStream::burnBits( S32 nbits ) { 
#if defined(__BITSTREAM_RANGE_CHECK__)
	if ( nbits + mBitNum > mMaxBitNum ) {
		mError = true;
		return;
	}
#endif

	mBitNum += ( nbits ); 
}

// Assumes byte boundaries on both sides
void BitStream::readAligned( void *buffer, int size ) {
#if defined(__BITSTREAM_RANGE_CHECK__)
	if ( ( size << 3 ) + mBitNum > mMaxBitNum ) {
		mError = true;
		return;
	}
#endif

	memcpy( buffer, mDataPtr + ( mBitNum >> 3 ), size );
	mBitNum += ( size << 3 );
}

void BitStream::readBits( S32 bitCount, void *bitPtr ) {
#if defined(__BITSTREAM_RANGE_CHECK__)
	if ( !bitCount )
		return;

	if( bitCount + mBitNum > mMaxBitNum ) {
		mError = true;
		return;
	}
#endif

	U8 *stPtr = ( mDataPtr + ( mBitNum >> 3 ) );
	S32 byteCount = ( bitCount + 7 ) >> 3;

	U8 *ptr = (U8 *) bitPtr;

	S32 downShift = ( mBitNum & 0x7 );
	S32 upShift = ( 8 - downShift );

	U8 curB = *stPtr;
	while ( byteCount-- ) {
		U8 nextB = *++stPtr;
		*ptr++ = ( curB >> downShift ) | ( nextB << upShift );
		curB = nextB;
	}

	mBitNum += ( bitCount );
}

bool BitStream::readFlag( ) {
#if defined(__BITSTREAM_RANGE_CHECK__)
	if( mBitNum >= mMaxBitNum ) {
		mError = true;
		return ( false );
	}
#endif
	
	S32 mask = ( 1 << ( mBitNum & 0x7 ) );
	return ( ( *( mDataPtr + ( mBitNum++ >> 3 ) ) & mask ) != 0 );
}

S32 BitStream::readInt( S32 bitCount ) {
	S32 ret = 0;
	readBits( bitCount, &ret );

#if defined(__USE_BIG_ENDIAN__)
	U32 i = (U32 )ret;
	ret = (S32 )
		(( (i & 0xff000000 ) >> 24 ) |
		 ( (i & 0x00ff0000 ) >>  8 ) |
		 ( (i & 0x0000ff00 ) <<  8 ) |
		 ( (i & 0x000000ff ) << 24 ));
#endif

	if ( bitCount == 32 )
		return ( ret );
	else
		ret &= ( ( 1 << bitCount ) - 1 );
	
	return ( ret );
}


F32 BitStream::readFloat( S32 bitCount ) {
	return ( readInt( bitCount ) / F32( ( 1 << bitCount ) - 1 ) );
}

F32 BitStream::readSignedFloat( S32 bitCount ) {
   return ( readInt( bitCount ) * 2 / F32( ( 1 << bitCount ) - 1 ) - 1.0f );
}

S32 BitStream::readSignedInt( S32 bitCount ) {
	if ( readFlag( ) )
		return ( -readInt( bitCount - 1 ) );
	else
		return (  readInt( bitCount - 1 ) );
}

// this is joke!
void BitStream::readNormalVector(S32 bitCount) {
	burnBits( bitCount + bitCount + 1 );
}

void BitStream::readString( char buf[256] ) {
	buf[ 0 ] = '\x0';

	if ( !mError )
		HuffmanProcessor::gHuffProcessor.readHuffBuffer( *this, buf );
}


void BitStream::setBit( S32 bitCount, bool set ) {
   if ( set )
      *( mDataPtr + ( bitCount >> 3 ) ) |=  ( 1 << ( bitCount & 0x7 ) );
   else
      *( mDataPtr + ( bitCount >> 3 ) ) &= ~( 1 << ( bitCount & 0x7 ) );
}

void BitStream::setBuffer( void *bufPtr, S32 size ) {
   mDataPtr = (U8 *) bufPtr;
   mBitNum = 0;
   mBufSize = size;
   mMaxBitNum = ( size << 3 ); 
   mError = false;
}

// Assumes byte boundaries on both sides
void BitStream::writeAligned( void *buffer, int size ) {
#if defined(__BITSTREAM_RANGE_CHECK__)
	if ( ( size << 3 ) + mBitNum > mMaxBitNum ) {
		mError = true;
		return;
	}
#endif

	memcpy( mDataPtr + ( mBitNum >> 3 ), buffer, size );
	mBitNum += ( size << 3 );
}

void BitStream::writeBits( S32 bitCount, const void *bitPtr ) {
#if defined(__BITSTREAM_RANGE_CHECK__)
	if( !bitCount )
		return;

	if( bitCount + mBitNum > mMaxBitNum ) {
		mError = true;
		return;
	}
#endif

	const U8 *ptr = (U8 *) bitPtr;
	U8 *stPtr = mDataPtr + ( mBitNum >> 3 );
	U8 *endPtr = mDataPtr + ( ( bitCount + mBitNum - 1 ) >> 3 );

	S32 upShift  = ( mBitNum & 0x7 );
	S32 downShift= ( 8 - upShift );
	U8 lastMask  = ( 0xFF >> ( 7 - ( ( mBitNum + bitCount - 1 ) & 0x7 ) ) );
	U8 startMask = ( 0xFF >> downShift );

	U8 curB = *ptr++;
	*stPtr = ( ( curB << upShift ) | ( *stPtr & startMask ) );

	stPtr++;
	while( stPtr <= endPtr ) {
		U8 nextB = *ptr++;
		*stPtr++ = ( ( curB >> downShift ) | ( nextB << upShift ) );
		curB = nextB;
	}
	*endPtr &= lastMask;

	mBitNum += ( bitCount );
}

void BitStream::writeInt( S32 val, S32 bitCount ) {
#if defined(__USE_BIG_ENDIAN__)
	U32 i = (U32 )val;
	val = (S32 )
		(( (i & 0xff000000 ) >> 24 ) |
		 ( (i & 0x00ff0000 ) >>  8 ) |
		 ( (i & 0x0000ff00 ) <<  8 ) |
		 ( (i & 0x000000ff ) << 24 ));
#endif

	writeBits( bitCount, &val );
}

void BitStream::writeFloat( F32 f, S32 bitCount ) {
   writeInt( (S32 )( f * ( ( 1 << bitCount ) - 1 ) ), bitCount );
}

void BitStream::writeSignedFloat( F32 f, S32 bitCount ) {
	writeInt( (S32 )( (f + 1) * .5 ) * ( ( 1 << bitCount ) - 1 ), bitCount );
}

void BitStream::writeSignedInt( S32 value, S32 bitCount ) {
	if( writeFlag( value < 0 ) )
		writeInt( -value, bitCount - 1 );
	else
		writeInt( value, bitCount - 1 );
}

bool BitStream::writeFlag( bool val ) {
#if defined(__BITSTREAM_RANGE_CHECK__)
	if( mBitNum >= mMaxBitNum ) {
		mError = true;
		return ( val );
	}
#endif

   if ( val )
      *( mDataPtr + ( mBitNum >> 3 ) ) |=  ( 1 << ( mBitNum & 0x7 ) );
   else
      *( mDataPtr + ( mBitNum >> 3 ) ) &= ~( 1 << ( mBitNum & 0x7 ) );
   mBitNum++;
   return ( val );
}

void BitStream::writeString( const char *string, S32 maxLen ) {
	if( !string )
		string = "";
	
	HuffmanProcessor::gHuffProcessor.writeHuffBuffer( *this, string, maxLen );
}

/*
	Huffman
*/

void HuffmanProcessor::buildTables( ) {
	mTablesBuilt = true;

	S32 i;

	// First, construct the array of wraps...
	mHuffNodeCount = 1;
	for (i = 0; i < 256; i++) {
		HuffLeaf &rLeaf = mHuffLeaves[i];

		rLeaf.pop = mCharFreqs[i];
		rLeaf.symbol = U8(i);
		rLeaf.numBits = 0;
		rLeaf.code = 0;
	}

	S32 currWraps = 256;
	HuffWrap *pWrap = new HuffWrap[ 256 ];
	for (i = 0; i < 256; i++)
		pWrap[i].set( &mHuffLeaves[i] );

	while ( currWraps != 1 ) {
		U32 min1 = 0xfffffffe, min2 = 0xffffffff;
		S32 index1 = -1, index2 = -1;

		for ( i = 0; i < currWraps; i++ ) {
			if ( pWrap[ i ].getPop() < min1 ) {
				min2   = min1;
				index2 = index1;

				min1   = pWrap[ i ].getPop();
				index1 = i;
			} else if ( pWrap[ i ].getPop() < min2 ) {
				min2   = pWrap[ i ].getPop();
				index2 = i;
			}
		}
	
		// Create a node for this...
		mHuffNodeCount++;
		HuffNode &rNode = mHuffNodes[ mHuffNodeCount ];
		rNode.pop    = pWrap[index1].getPop() + pWrap[index2].getPop();
		rNode.index[0] = determineIndex(pWrap[index1]);
		rNode.index[1] = determineIndex(pWrap[index2]);

		S32 mergeIndex = ( index1 > index2 ) ? index2 : index1;
		S32 nukeIndex  = ( index1 > index2 ) ? index1 : index2;
		pWrap[ mergeIndex ].set( &rNode );

		if ( index2 != ( currWraps - 1 ) )
			pWrap[ nukeIndex ] = pWrap[ currWraps - 1 ];

		currWraps--;
	}

	// Ok, now we have one wrap, which is a node.  we need to make sure that this
	//  is the first node in the node list.
	mHuffNodes[0] = *( pWrap[0].pNode );
	delete[] pWrap;

	U32 code = 0;
	BitStream bs( &code, 4 );
	generateCodes( bs, 0, 0 );
}

void HuffmanProcessor::generateCodes( BitStream &rBS, S32 index, S32 depth ) {
	if ( index < 0 ) { 
		// leaf node, copy the code in, and back out...
		HuffLeaf& rLeaf = mHuffLeaves[ -(index + 1) ];
		memcpy( &rLeaf.code, rBS.mDataPtr, sizeof( rLeaf.code ) );
		rLeaf.numBits = depth;
	} else {
		HuffNode& rNode = mHuffNodes[index];

		S32 pos = rBS.getBitPos();
		rBS.writeFlag( false );
		generateCodes( rBS, rNode.index[0], depth + 1 );

		rBS.setBitPos( pos );
		rBS.writeFlag( true );
		generateCodes( rBS, rNode.index[1], depth + 1 );

		rBS.setBitPos( pos );
	}
}

S16 HuffmanProcessor::determineIndex( HuffWrap &rWrap ) {
	if ( rWrap.pLeaf != NULL ) {
		return -(S16 )( ( rWrap.pLeaf - mHuffLeaves ) + 1 );
	} else {
		return (S16 )( rWrap.pNode - mHuffNodes );
	}
}

void HuffmanProcessor::readHuffBuffer( BitStream &stream, char* buffer ) {
	if ( !mTablesBuilt )
		buildTables( );

	bool compressed = ( stream.readFlag( ) );
	S32 len = stream.readInt( 8 ); // stored as 2^8 so it can never exceed 255!

	if ( compressed ) {
		for ( S32 i = 0; i < len; i++ ) {
			S32 index = 0;
			// Flag indicates which node to follow
			while ( index >= 0 )
				index = mHuffNodes[index].index[ (U8)stream.readFlag( ) ];
			
			// Leaves have a negative index
			buffer[ i ] = ( mHuffLeaves[ -(index+1) ].symbol );
		}
	} else {
		stream.readBits( len << 3, buffer );
	}

	buffer[ len ] = '\0';
}

bool HuffmanProcessor::writeHuffBuffer( BitStream &stream, const char *buffer, S32 maxLen ) {
	if ( buffer == NULL ) {
		stream.writeFlag( false );
		stream.writeInt( 0, 8 );
		return ( true );
	}

	if ( !mTablesBuilt ) 
		buildTables( );

	S32 len = ( buffer ) ? (S32 )strlen( buffer ) : 0;
	if ( len > maxLen )
		len = maxLen;

	S32 numBits = 0;
	S32 i;

	// Sum the bits first
	for ( i = 0; i < len; i++ )
		numBits += ( mHuffLeaves[ (unsigned char)buffer[ i ] ].numBits );

	// If it's larger than just writing the string uncompressed
	if ( numBits >= ( len * 8 ) ) {
		stream.writeFlag( false );
		stream.writeInt( len, 8 );
		stream.write( (void *)buffer, len );
	} else {
		stream.writeFlag( true );
		stream.writeInt( len, 8 );
		for ( i = 0; i < len; i++ ) {
			HuffLeaf& rLeaf = mHuffLeaves[ ( (U8)buffer[i] ) ];
			stream.writeBits( rLeaf.numBits, &rLeaf.code );
		}
	}

	return ( true );
}


const U32 HuffmanProcessor::mCharFreqs[256] = {
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x014a, 0x0016, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0afa, 0x0045, 0x0001, 0x001c, 0x0001, 0x003b, 0x0004, 0x003f,
	0x0005, 0x0008, 0x0001, 0x0001, 0x0010, 0x0042, 0x022b, 0x0004,
	0x018c, 0x0196, 0x00bf, 0x0077, 0x0020, 0x0035, 0x001d, 0x0011,
	0x0024, 0x0022, 0x0051, 0x0002, 0x008f, 0x0004, 0x008f, 0x0028,
	0x0001, 0x0092, 0x007f, 0x002e, 0x007c, 0x0115, 0x0048, 0x0089,
	0x003f, 0x0081, 0x000a, 0x000e, 0x0073, 0x00f8, 0x007c, 0x0026,
	0x00bb, 0x0003, 0x0097, 0x0137, 0x0151, 0x000e, 0x000d, 0x0010,
	0x0038, 0x0099, 0x0002, 0x0001, 0x0003, 0x0001, 0x0001, 0x00d4,
	0x0001, 0x082c, 0x015a, 0x02e2, 0x03e3, 0x0b3a, 0x02bf, 0x025f,
	0x0288, 0x0612, 0x014a, 0x0133, 0x04da, 0x02e1, 0x05ff, 0x06b3,
	0x0234, 0x0005, 0x06f1, 0x047f, 0x05bf, 0x03d5, 0x0199, 0x022b,
	0x003d, 0x0119, 0x0021, 0x0001, 0x0001, 0x0001, 0x0045, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
};

