{
 Small utility to extract the textures from Tribes Ascend content files

 They seem to be simply stored ZIP compressed for each mipmap level as DXT1 compressed textures.
 The header file follows a very simple format.
 Should work for most stuff I think

 Send me beer at jeppe@j-software.dk
}
program tribestexture;

{$mode objfpc}{$H+}

uses classes, sysutils, getopts,
     dateutils,
     zstream;

type
 TFragment = record
  Offset,
  Length,
  Bytes: longword;
 end;

 TTexture = record
  Name: string;
  Flags: word;
  w,h: word;
  Fragments: array of TFragment;
 end;

 TColor565 = word;

 TColor = packed record
  case integer of
   0: (b,g,r,a: byte);
   2: (x: array[0..3] of byte);
 end;

 PDXT1Tile = ^TDXT1Tile;
 TDXT1Tile = packed record
  c0, c1: TColor565;
  Pixels: array[0..3] of byte; // 4x4 matrix of 2 bits
 end;

 PDXT5Tile = ^TDXT5Tile;
 TDXT5Tile = packed record
  a0, a1: byte;
  alphas: array[0..5] of byte;

  c0, c1: TColor565;
  Pixels: array[0..3] of byte; // 4x4 matrix of 2 bits
 end;

 PDXT3Tile = ^TDXT3Tile;
 TDXT3Tile = packed record
  alphas: array[0..7] of byte;

  c0, c1: TColor565;
  Pixels: array[0..3] of byte; // 4x4 matrix of 2 bits
 end;

 TTGAHeader = packed record
  IDLength: Byte;
  CMapType: Byte;
  ImageType: Byte;
  CMapSpec: packed record
   FirstEntryIndex: Word;
   Length: Word;
   EntrySize: Byte;
  end;
  ImgSpec: packed record
   Xorg: Word;
   Yorg: Word;
   Width: Word;
   Height: Word;
   Depth: Byte;
   Desc: Byte;
  end;
 end;

var HdrPath: string = 'texture.cache.hdr.rtc';
    DataPath: string = 'texture.cache.data.rtc';

    List: boolean = false;
    ExportFile: string = '';
    ExportPath: string = 'texture.tga';
    ExportAll: boolean = false;

    Textures: array of TTexture;

    DiscardAlpha: boolean = false;

function ReadTexture(s: TStream): TTexture;
var cnt: Cardinal;
    typ: Byte;
    i: longint;
begin
   result.Name := s.ReadAnsiString;
   result.Flags := s.ReadWord;
   result.w := s.ReadWord;
   result.h := s.ReadWord;

   cnt := s.ReadDWord;

   setlength(result.Fragments, cnt);

   for i := 0 to cnt-1 do
   begin
      typ := s.ReadByte;
      if typ <> 1 then
         raise exception.CreateFmt('Got type: %d at pos %d: %s', [typ, s.Position-1, result.name]);

      if typ = 1 then
      begin
         result.Fragments[i].Offset := s.ReadDWord;
         result.Fragments[i].Length := s.ReadDWord;
         result.Fragments[i].Bytes  := s.ReadDWord;
      end;
   end;
end;

procedure LoadTextureInfo;
var fs: TStream;
    cnt, i: longint;
begin
   fs := TFileStream.Create(HdrPath, fmOpenRead);
   try
      Cnt := fs.ReadDWord;

      SetLength(textures, cnt);

      for i := 0 to cnt-1 do
         Textures[i] := ReadTexture(fs);
   finally
      fs.Free;
   end;
end;

procedure ParseOptions;
var theopts: array[1..8] of TOption;
    c: Char;
    opt: Longint;
begin
   if Paramcount < 1 then
   begin
      WriteLn('Usage: tribestexture [-l] [-a] [-n] [-h <file.rtc>] [-d <file.rtc>] [-e <name>] [-o <imagename>]');
      writeln(' -l : List all names');
      writeln(' -h : Header filename');
      writeln(' -d : Data filename');
      writeln(' -e : Export named texture');
      writeln(' -n : Discard alpha channel');
      writeln(' -o : Filename of outputted TGA texture. Outputted in highest resolution.');
      writeln(' -a : Export all.');
      halt;
   end;

   with theopts[1] do
   begin
      name:='header';
      has_arg:=1;
      flag:=nil;
      value:=#0;
   end;
   with theopts[2] do
   begin
      name:='data';
      has_arg:=1;
      flag:=nil;
      value:=#0;
   end;
   with theopts[3] do
   begin
      name:='list';
      has_arg:=0;
      flag:=nil;
      value:=#0;
   end;
   with theopts[4] do
   begin
      name:='export';
      has_arg:=1;
      flag:=nil;
      value:=#0;
   end;
   with theopts[5] do
   begin
      name:='output';
      has_arg:=1;
      flag:=nil;
      value:=#0;
   end;
   with theopts[6] do
   begin
      name:='noalpha';
      has_arg:=0;
      flag:=nil;
      value:=#0;
   end;
   with theopts[7] do
   begin
      name:='all';
      has_arg:=0;
      flag:=nil;
      value:=#0;
   end;
   with theopts[8] do
   begin
      name:='';
      has_arg:=0;
      flag:=nil;
      value:=#0;
   end;

   repeat
      c := GetLongOpts('ad:e:h:lo:n', @theopts[1], opt);

      case c of
         'h': HdrPath := OptArg;
         'd': DataPath := OptArg;
         'l': List := true;
         'e': ExportFile := OptArg;
         'o': ExportPath := OptArg;
         'n': DiscardAlpha := true;
         'a': ExportAll := true;
      end;
   until c=EndOfOptions;
end;

function RGB565ToColor(const c: TColor565): TColor;
begin
   result.b := (c and $1F) shl 3;
   result.g := ((c shr 5) and $3F) shl 2;
   result.r := ((c shr 11) and $1F) shl 3;
   result.a := 255;
end;

procedure ExportTex(Format, Offset, Length, Bytes, W,H: longint);
var fs, ms, zs,
    ImgStream: TStream;
    hdr: word;
    i,i2, x,y,
    w4,h4,
    cnt: longint;
    LUT: byte;

    header: TTGAHeader;
    Bits: array of byte;
    Pixels: array of array of TColor;

    c: array[0..3] of TColor;
    a: array[0..7] of byte;

   procedure PrepareDXT1(Index: longint);
   begin
      c[0] := RGB565ToColor(PDXT1Tile(@Bits[index])^.c0);
      c[1] := RGB565ToColor(PDXT1Tile(@Bits[index])^.c1);

      if PDXT1Tile(@Bits[index])^.c0 > PDXT1Tile(@Bits[index])^.c1 then
      begin
         c[2].r := (2*longint(c[0].r)+longint(c[1].r)) div 3;
         c[2].g := (2*longint(c[0].g)+longint(c[1].g)) div 3;
         c[2].b := (2*longint(c[0].b)+longint(c[1].b)) div 3;
         c[2].a := 255;

         c[3].r := (longint(c[0].r)+2*longint(c[1].r)) div 3;
         c[3].g := (longint(c[0].g)+2*longint(c[1].g)) div 3;
         c[3].b := (longint(c[0].b)+2*longint(c[1].b)) div 3;
         c[3].a := 255;
      end
      else
      begin
         c[2].r := (longint(c[0].r)+longint(c[1].r)) div 2;
         c[2].g := (longint(c[0].g)+longint(c[1].g)) div 2;
         c[2].b := (longint(c[0].b)+longint(c[1].b)) div 2;
         c[2].a := 255;

         c[3].r := 0;
         c[3].g := 0;
         c[3].b := 0;
         if DiscardAlpha then
            c[3].a := 255
         else
            c[3].a := 0;
      end;
   end;

   procedure PrepareDXT23(ColorOffset: longint);
   begin
      c[0] := RGB565ToColor(PDXT1Tile(@Bits[ColorOffset])^.c0);
      c[1] := RGB565ToColor(PDXT1Tile(@Bits[ColorOffset])^.c1);

      c[2].r := (2*longint(c[0].r)+longint(c[1].r)) div 3;
      c[2].g := (2*longint(c[0].g)+longint(c[1].g)) div 3;
      c[2].b := (2*longint(c[0].b)+longint(c[1].b)) div 3;
      c[2].a := 255;

      c[3].r := (longint(c[0].r)+2*longint(c[1].r)) div 3;
      c[3].g := (longint(c[0].g)+2*longint(c[1].g)) div 3;
      c[3].b := (longint(c[0].b)+2*longint(c[1].b)) div 3;
      c[3].a := 255;
   end;

   function MixA(w0, w1, d: longint): byte;
   begin
      result := (longint(a[0])*w0+longint(a[1])*w1) div d;
   end;

   procedure PrepareDXT5(Index: longint);
   begin
      PrepareDXT23(Index+8);

      a[0] := PDXT5Tile(@Bits[index])^.a0;
      a[1] := PDXT5Tile(@Bits[index])^.a1;

      if a[0] > a[1] then
      begin
         a[2] := MixA(6,1,7);
         a[3] := MixA(5,2,7);
         a[4] := MixA(4,3,7);
         a[5] := MixA(3,4,7);
         a[6] := MixA(2,5,7);
         a[7] := MixA(1,6,7);
      end
      else
      begin
         a[2] := MixA(4,1,5);
         a[3] := MixA(3,2,5);
         a[4] := MixA(2,3,5);
         a[5] := MixA(1,4,5);
         a[6] := 0;
         a[7] := 255;
      end;
   end;

   procedure DXT1(var idx: longint; lx,ly: longint);
   var x,y: longint;
       lut: byte;
   begin
      PrepareDXT1(idx);

      for y := 0 to 3 do
      begin
         LUT := PDXT1Tile(@Bits[idx])^.Pixels[y];
         for x := 0 to 3 do
         begin
            Pixels[ly+y][lx+x] := c[LUT and 3];
            LUT := LUT shr 2;
         end;
      end;

      inc(idx, sizeof(TDXT1Tile));
   end;

   procedure DXT3(var idx: longint; lx,ly: longint);
   var x,y: longint;
       lut: byte;

      function GetAlpha(ofs: longint): longint;
      begin
         result := (PDXT3Tile(@Bits[idx])^.alphas[ofs div 2] shr (ofs and 1)) shl 4;
      end;

   begin
      PrepareDXT23(idx+8);

      for y := 0 to 3 do
      begin
         LUT := PDXT3Tile(@Bits[idx])^.Pixels[y];
         for x := 0 to 3 do
         begin
            Pixels[ly+y][lx+x] := c[LUT and 3];
            if DiscardAlpha then
               Pixels[ly+y][lx+x].a := 255
            else
               Pixels[ly+y][lx+x].a := GetAlpha(x+y*4);
            LUT := LUT shr 2;
         end;
      end;

      inc(idx, sizeof(TDXT3Tile));
   end;

   procedure DXT5(var idx: longint; lx,ly: longint);
   var x,y: longint;
       lut: byte;

      function GetAlpha(ofs: longint): longint;
      begin
         result := (pword(ptruint(@PDXT5Tile(@Bits[idx])^.alphas)+ofs div 3)^ shr (ofs mod 3)) and $7;
      end;

   begin
      PrepareDXT5(idx);

      for y := 0 to 3 do
      begin
         LUT := PDXT5Tile(@Bits[idx])^.Pixels[y];
         for x := 0 to 3 do
         begin
            Pixels[ly+y][lx+x] := c[LUT and 3];
            if DiscardAlpha then
               Pixels[ly+y][lx+x].a := 255
            else
               Pixels[ly+y][lx+x].a := a[GetAlpha(x+y*4)];
            LUT := LUT shr 2;
         end;
      end;

      inc(idx, sizeof(TDXT5Tile));
   end;

begin
   w4 := w div 4;
   h4 := h div 4;

   fs := TFileStream.Create(DataPath, fmOpenRead);
   try
      fs.Seek(Offset, soFromBeginning);

      hdr := fs.ReadWord;
      if hdr <> $9C78 then raise exception.CreateFmt('Wrong header: %4x', [hdr]);

      ms := TMemoryStream.Create;
      try
         ms.CopyFrom(fs, length);
         ms.Seek(0, soFromBeginning);

         zs := Tdecompressionstream.Create(ms, true);
         try
            SetLength(bits, bytes);

            setlength(pixels, h);
            for i := 0 to h-1 do
               SetLength(Pixels[i], w);

            // Read the ZIP compressed data
            zs.ReadBuffer(Bits[0], bytes);

            ImgStream := TFileStream.Create(ExportPath, fmCreate or fmOpenWrite);
            try
               // Decompress tiles
               case format of
                  $205:
                     begin
                        cnt := 0;
                        for i := 0 to h4-1 do
                           for i2 := 0 to w4-1 do
                              DXT1(cnt, i2*4, i*4);
                     end;
                  $207:
                     begin
                        cnt := 0;
                        for i := 0 to h4-1 do
                           for i2 := 0 to w4-1 do
                              DXT5(cnt, i2*4, i*4);
                     end;
               else
                  raise exception.CreateFmt('Unsupported format(%x)', [format]);
               end;

               case format of
                  $205,
                  $207:
                     begin
                        // Write a header
                        FillChar(header, sizeof(header), 0);
                        header.CMapType := 0;
                        header.ImageType := 2;

                        header.ImgSpec.Xorg := 0;
                        header.ImgSpec.Yorg := 0;
                        header.ImgSpec.Width := w;
                        header.ImgSpec.Height := h;
                        header.ImgSpec.Depth := 32;
                        header.ImgSpec.Desc := $00;

                        ImgStream.WriteBuffer(header, sizeof(header));

                        // Save in reverse order. Most readers don't handle the inverse direction bit
                        for y := h4-1 downto 0 do
                           for i2 := 3 downto 0 do
                              ImgStream.WriteBuffer(Pixels[y*4+i2][0], w*4);
                     end;
               end;
            finally
               ImgStream.Free;
            end;
         finally
            zs.Free;
         end;
      finally
         ms.Free;
      end;
   finally
      fs.Free;
   end;
end;

var i: longint;
    start: TDateTime;
begin
   ParseOptions;
   LoadTextureInfo;

   if List then
      for i := 0 to high(Textures) do
         WriteLn(textures[i].Name,', ', textures[i].w,', ', textures[i].h, ' - ', inttohex(textures[i].Flags, 8), ' - ', IntToHex(textures[i].Fragments[0].Bytes, 8));

   if ExportFile <> '' then
   begin
      for i := 0 to high(textures) do
      begin
         if textures[i].Name = ExportFile then
         begin
            try
               ExportTex(Textures[i].Flags, Textures[i].Fragments[0].Offset, Textures[i].Fragments[0].Length, Textures[i].Fragments[0].Bytes, Textures[i].w, Textures[i].h);
            except
               on e: exception do
                  writeln('Error while exporting: ', e.Message);
            end;

            exit;
         end;
      end;
      Writeln('No texture found with name: "', ExportFile, '"');
   end;

   if ExportAll then
   begin
      start := now;

      for i := 0 to high(textures) do
      begin
         writeln(i+1,' out of ', length(textures), '(',i*100 div high(textures), ' %) Approx time left: ', (MilliSecondsBetween(now, start)*length(textures) div (i+1) - MilliSecondsBetween(now, start)) div 60000, ' minutes');

         if length(Textures[i].Fragments) = 0 then
            continue;

         ExportPath := textures[i].Name+'.tga';
         try
            ExportTex(Textures[i].Flags, Textures[i].Fragments[0].Offset, Textures[i].Fragments[0].Length, Textures[i].Fragments[0].Bytes, Textures[i].w, Textures[i].h);
         except
            on e: exception do
               writeln(StdErr, 'Error while exporting(',textures[i].Name,'): ', e.Message);
         end;
      end;
   end;
end.

