//
// AI support functions.
//

//
// This function creates an AI player using the supplied group of markers 
//    for locations.  The first marker in the group gives the starting location 
//    of the the AI, and the remaining markers specify the path to follow.  
//
// Example call:  
// 
//    createAI( guardNumberOne, "MissionGroup\\Teams\\team0\\guardPath", larmor );
//
function createAI( %aiName, %markerGroup, %armorType, %name )
{
   %group = nameToID( %markerGroup );
   
   if( %group == -1 || Group::objectCount(%group) == 0 )
   {
      dbecho(2, %aiName @ "Couldn't create AI: " @ %markerGroup @ " empty or not found." );
      return -1;
   }
   else
   {
      %spawnMarker = Group::getObject(%group, 0);
      %spawnPos = GameBase::getPosition(%spawnMarker);
      %spawnRot = GameBase::getRotation(%spawnMarker);

      if( AI::spawn( %aiName, %armorType, %spawnPos, %spawnRot, %name, "male2" ) != "false" )
      {
         // The order number is used for sorting waypoints, and other directives.  
         %orderNumber = 100;
         
         for(%i = 1; %i < Group::objectCount(%group); %i = %i + 1)
         {
             
            %spawnMarker = Group::getObject(%group, %i);
            %spawnPos = GameBase::getPosition(%spawnMarker);
            
            AI::DirectiveWaypoint( %aiName, %spawnPos, %orderNumber );
            
            %orderNumber += 100;
         }
      }
      else{
         dbecho( 2, "Failure spawning: " @ %aiName );
      }
   }
}

//----------------------------------
//Training::setupAI()
//----------------------------------   
function AI::setupAI(%key, %team)
{
   //if there is no key then they don't exist yet
   if(%key == "")
   {
      %aiFound = 0;
      for( %T = 0; %T < 8; %T++ )
      {
         %groupId = nameToID("MissionGroup\\Teams\\team" @ %T @ "\\AI" );
         if( %groupId != -1 )
         {
            %teamItemCount = Group::objectCount(%groupId);
            if( %teamItemCount > 0 )
            {
               AI::initDrones(%T, %teamItemCount);
               %aiFound += %teamItemCount;
            }
         }
      }
      if( %aiFound == 0 )
         dbecho(2, "No drones exist...");
      else
         dbecho(2, %aiFound @ " drones installed..." );
   }
   else     //respawning dead AI with original name and path
   {
      %group = nameToID("MissionGroup\\Teams\\team" @ %team @ "\\AI\\" @ %key);
      %num = Group::objectCount(%group);
      createAI(%key, %group, larmor, %key);
      %aiId = AI::getId(%key);
      GameBase::setTeam(%aiId, %team);
      AI::setWeapons(%key);
   }		
}

//------------------------------
//AI::setWeapons()
//------------------------------
function AI::setWeapons(%aiName)
{
	%aiId = AI::getId(%aiName);
   Player::setItemCount(%aiId, blaster, 1);
   Player::setItemCount(%aiId, disclauncher, 1);
   Player::setItemCount(%aiId, chaingun, 1);
   Player::setItemCount(%aiId, discammo, 50);
   Player::setItemCount(%aiId, bulletAmmo, 500);
   Player::mountItem(%aiId, blaster, 0);
   AI::SetVar(%aiName, triggerPct, 0.03 );
   AI::setVar(%aiName, iq, 60 );
   AI::setVar(%aiName, attackMode, 1);
   AI::setAutomaticTargets( %aiName );
   ai::callbackPeriodic(%aiName, 5, periodicWeaponChange);
}


//-----------------------------------
//AI::weapongchange()
//-----------------------------------
function periodicWeaponChange(%aiName)
{
	dbecho(2, %aiName @ " thinking....");
   %aiId = AI::getId(%aiName);
   %curTarget = ai::getTarget( %aiName );
   
   if(%curTarget == -1)
   {
   	return;
   }
   else
   {   
      dbecho(2, %aiName @ " target: " @ %curTarget);	
      
      %targLoc = GameBase::getPosition(Client::getOwnedObject(%curTarget));
      %aiLoc = GameBase::getPosition(Client::getOwnedObject(%aiId));
      %targetDist = Vector::getDistance(%aiLoc, %targLoc);
      dbecho(2, "distance to target: " @ %targetDist @ 
                     " targetPosition: " @ targLoc @ " aiLocation: " @ %aiLoc);
                     
      if(%targetDist > 100)
      {
      	Player::mountItem(%aiId, blaster, 0);
         AI::SetVar(%aiName, triggerPct, 0.03 );
      }   
      else
      {
         dbecho(2, "checking for target jet...");
         dbecho(2, "jetting? " @ Player::isJetting(%curTarget));
         if(Player::isJetting(%curTarget))
         {	
            Player::mountItem(%aiId, Chaingun, 0);
            AI::SetVar(%aiName, triggerPct, 0.9 );
         }
         else
         {
            Player::mountItem(%aiId, disclauncher, 0);  
            AI::SetVar(%aiName, triggerPct, 0.03 );
         }   
      }
   }
}

//-----------------------------------
//AI::initDrones()
//-----------------------------------
function AI::initDrones(%team, %numAi)
{
	dbecho( 2, "spawning team " @ %team @ " ai...");
   for(%guard = 0; %guard < %numAi; %guard++)
   {
      //check for internal data
      %tempSet = 	nameToID("MissionGroup\\Teams\\team" @ %team @ "\\AI");
      %tempItem = Group::getObject(%tempSet, %guard);
      %aiName = Object::getName(%tempItem);
      
      %set = nameToID("MissionGroup\\Teams\\team" @ %team @ "\\AI\\" @ %aiName);
      %numPts = Group::objectCount(%set);
      
      if(%numPts > 0)
      {
         createAI(%aiName, %set, larmor, %aiName);
         %aiId = AI::getId( %aiName );
         GameBase::setTeam(%aiId, %team);
         AI::setVar( %aiName,  iq,  60 );
         AI::setVar( %aiName,  attackMode, 1);
      	AI::setWeapons(%aiName);
    	}
      else
         dbecho( 2, "no info to spawn ai...");
   }
}


//------------------------------------------------------------------
//functions to test and move AI players.
//
//------------------------------------------------------------------

//
//This function will spawn an AI player about 5 units away from the 
//player that is passed to the function(%commandIssuer).
//
//
$numAI = 0;
function AI::helper(%aiName, %armorType, %commandIssuer)
{
   %spawnMarker = GameBase::getPosition(%commandIssuer);
   %xPos = getWord(%spawnMarker, 0) + floor(getRandom() * 15);
   %yPos = getword(%spawnMarker, 1) + floor(getRandom() * 10);
   %zPos = getWord(%spawnMarker, 2) + 2;
   %rPos = GameBase::getRotation(%commandIssuer);
   
   dbecho(2, "Spawning AI helper at position " @ %xPos @ " " @ %yPos @ " " @ %zPos);
   dbecho(2, "Current Issuer rotation: " @ %rPos);
      
   %aiSpawnPos = %xPos @ "  " @ %yPos @ "  " @ %zPos;
   %newName = %aiName @ $numAI;
   $numAI++;
   Ai::spawn(%newName, %armorType, %aiSpawnPos, %rPos);
   return ( %newName );
}

//
//This function will move an AI player to the position of an object
//that the players LOS is hitting(terrain included). Must be `	within 50 units.
//
//
function AI::moveToLOS(%aiName, %commandIssuer) 
{
   %issuerRot = GameBase::getRotation(%commandIssuer);
   %playerObj = Client::getOwnedObject(%commandIssuer);
   %playerPos = GameBase::getPosition(%commandIssuer);
      
   //check within max dist
   if(GameBase::getLOSInfo(%playerObj, 100, %issuerRot))
   { 
      %newIssuedVec = $LOS::position;
	  %distance = Vector::getDistance(%playerPos, %newIssuedVec);
	  dbecho(2, "Command accepted, AI player(s) moving....");
	  dbecho(2, "distance to LOS: " @ %distance);
	  AI::DirectiveWaypoint( %aiName, %newIssuedVec, 1 );
   }
   else
      dbecho(2, "Distance to far.");
      
   dbecho(2, "LOS point: " @ $LOS::position);
}

//This function will move an AI player to a position directly in front of
//the player passed, at a distance that is specified.
function  AI::moveAhead(%aiName, %commandIssuer, %distance) 
{
   
   %issuerRot = GameBase::getRotation(%commandIssuer);
   %commPos  = GameBase::getPosition(%commandIssuer);
   dbecho(2, "Commanders Position: " @ %commPos);
   
   //get commanders x and y positions
   %comm_x = getWord(%commPos, 0);
   %comm_y = getWord(%commPos, 1);
   
   //get offset x and y positions
   %offSetPos = Vector::getFromRot(%issuerRot, %distance);
   %off_x = getWord(%offSetPos, 0);
   %off_y = getWord(%offSetPos, 1);
   
   //calc new position
   %new_x = %comm_x + %off_x;
   %new_y = %comm_y + %off_y;
   %newPos = %new_x  @ " " @ %new_y @ " 0";
  
   //move AI player
   dbecho(2, "AI moving to " @ %newPos);
   AI::DirectiveWaypoint(%aiName, %newPos, 1);
}  

//
// OK, this is the complete command callback - issued for any command sent
//    to an AI. 
//
function AI::onCommand ( %name, %commander, %command, %waypoint, %targetId, %cmdText, 
         %cmdStatus, %cmdSequence )
{
   %aiId = AI::getId( %name );
   %T = GameBase::getTeam( %aiId );
   %groupId = nameToID("MissionGroup\\Teams\\team" @ %T @ "\\AI\\" @ %name ); 
  	%nodeCount = Group::objectCount( %groupId );
   dbecho(2, "checking drone information...." @ " number of nodes: " @ %nodeCount);
   dbecho(2, "AI id: " @ %aiId @ " groupId: " @ %groupId);
   
   if($SinglePlayer || %nodeCount == 1)
   {
	   if( %command == 2 || %command == 1 )
	   {
	      // must convert waypoint location into world location.  waypoint location
	      //    is given in range [0-1023, 0-1023].  
	      %worldLoc = WaypointToWorld ( %waypoint );
	      AI::DirectiveWaypoint( %name, %worldLoc, 125 );
	      dbecho ( 2, %name @ " IS PROCEEDING TO LOCATION " @ %worldLoc );
	   }
	   dbecho( 2, "AI::OnCommand() issued to  " @ %name @ "  with parameters: " );
	   dbecho( 3, "Cmdr:        " @ %commander );
	   dbecho( 3, "Command:     " @ %command );
	   dbecho( 3, "Waypoint:    " @ %waypoint );
	   dbecho( 3, "TargetId:    " @ %targetId );
	   dbecho( 3, "cmdText:     " @ %cmdText );
	   dbecho( 3, "cmdStatus:   " @ %cmdStatus );
	   dbecho( 3, "cmdSequence: " @ %cmdSequence );
   }
   else
   	return;   
}


// Play the given wave file FROM %source to %DEST.  The wave name is JUST the basic wave
//    name without voice base info (which it will grab for you from the source client Id).  
//
// Example:
//    Ai::soundHelp( 2051, 2049, cheer3 );
//
function Ai::soundHelper( %sourceId, %destId, %waveFileName )
{
   %wName = strcat( "~w", Client::getVoiceBase( %sourceId ) );
   %wName = strcat( %wName, ".w" );
   %wName = strcat( %wName, %waveFileName );
   %wName = strcat( %wName, ".wav" );
   
   dbecho( 2, "Trying to play " @ %wName );
   
   Client::sendMessage( %destId, 0, %wName );
}


// Default periodic callback.  [Note by default it isn't called unless a frequency 
//    is set up using AI::CallbackPeriodic().  Type in that command to see how 
//    it works].  
function AI::onPeriodic( %aiName )
{
   dbecho( 2, "onPeriodic() called with " @ %aiName );
}


//
//incomplete function that is only going to 
//be used for testing.
//
//    function AI::hunt(%commander) 
//    {
//      %k = 0;
//      
//      %markerPos = GameBase::getPosition(%commander);
//      
//      %set = newObject("set",SimSet);
//      %mask = $SimPlayerObjectType;
//      %targets = containerBoxFillSet(%set,%mask,%markerPos,100,100,20,1);
//      echo("Number of targets:" @ %targets);
//      
//      for(%i = 0; %i < %targets; %i++)
//      {
//        %lockedObj = Group::getObject(%set, %i);
//        echo("Object locked:" @ %lockedObj);
//        echo("checking for continuity...");
//    	
//    	if(Player::getClient(%lockedObj) == -1)
//    	  %lockedObj = Group::getObject(%set, %i++);
//      }
//      echo("finished with set so deleting...");
//      deleteObject(%set);
//    
//      echo("Target = " @ %lockedObj);
//      %target = Player::getClient(%lockedObj);
//      AI::attack(%commander, %target);
//    
//    }

function AI::onDroneKilled(%aiName)
{
   if( ! $SinglePlayer )
   {
      %aiId = AI::getId(%aiName);
      %team = GameBase::getTeam(%aiId);
      dbecho(2, "AI Id: " @ %aiId);
      
      // Trying a little longer delay -
      schedule("AI::setupAI(" @ %aiName @ ", " @ %team @ ");", 7 );
      
   	// %wName = "w_taunt10";
      // AI::soundHelper( %aiId, 2051, %wName );
   }
   else
   {
      // just in case:
      dbecho( 2, "Non training callback called from Training" );
   }
}

//these AI function callbacks can be very useful!
function AI::onTargetLOSAcquired(%aiName, %idNum)
{
}

function AI::onTargetDied(%aiName, %idNum)
{
}                                 

function AI::onTargetLOSLost(%aiName, %idNum)
{
}

function AI::onTargetLOSRegained(%aiName, %idNum)
{
}
