// -- ObjectiveTrak -- v2.0 beta 2 -------------------------------------------------------------------------
// Written by |HH|Crunchy (Jonathan Slark)
// Email: jonslark@barrysworld.co.uk
//
// *** Requires the Presto Pack v0.93 or later ***  http://www.planetstarsiege.com/presto/
//
// Scripting tool that tracks the status of the objectives in missions.  The script currently tracks
// objectives from C&H, D&D and F&R mission types.  Note however you can get these types of objective in
// other mission types, the tracking isn't dependant on mission type.  For instance C&H (tower) type
// objectives can be found scatered through all the mission types.  In addition the C&H traking actually
// works in multi-team missions!  A caveat is that Presto's team name tracking only works for two teams.  I'm
// hoping to add in support for CTF and more importantly multi-team CTF.  Take a look at the new events and
// functions ObjectiveTrak defines listed below.  For an example of the use of the script check out my
// Objective HUD.
//
// The script will learn about new objectives from the messages sent to the client about them and will then
// save  this information so that the next time you play that mission it will then know about the objectives
// right from the start of the level!  There is a pack of files available that has mission
// information, usuable in a script, for all the missions that come with Tribes.
//
// The current implemetation assigns each objective unique ID's.  The ID's are intergers counting up from
// zero to one less than the number returned by the Objective::Number() function (see below).  When Presto
// has finished his implementation of lists then I may update this script to take advantage of this.
//
// The script has the nice side effect of being able to help TeamTrak learn the names of the teams.  In a C&H
// type mission for example this is the only way to find out the team names.
//
// This script uses plugins to do the work with different objective types.  A scripter could write a plugin
// for a new objective type.  The plugin would be an extra script that the end user executes in addition to
// this script.  See the bottom part of this file for more information.
//
// Example:
//
//	To check the status of all the objectives you could use a simple loop to run through them:
//
//		for(%id = 0; id < Objective::Number(); id++)
//		{
//			%status = Objective::GetStatus(%id);
//			%type = Objective::GetType(%id);
//			if(%type == $Objective::CandH)
//			{
//				if(%status == $Objective::Enemy)
//					(do something useful regarding enemy having the objective)
//				else if...
//			}
//		}
//
//
// New events:
//
//	eventObjectiveReset
//
//		Objectives variables have just been reset between missions or when first connected.  There will be no
//		objectives being tracked.  We won't know about them until they are loaded or discovered about mid
//		game.
//
//	eventObjectiveLoaded(%num)
//
//		Called when the objectives are loaded off disk, the number loaded is returned.  If they can't be
//		loaded the objectives will be added when they are taken or lost.  After each mission any new
//		objectives will be saved to the config directory so that the next time you play that level the
//		script will know about the objectives.  There is a file pack available that contains objective
//		information for all the base missions.  These are used if found in the config\Mission directory, any
//		files in the config directory will take precedence however.  If only a partial list was saved out
//		then the script will still learn about new objectives when they are claimed.
//
//	eventObjectiveMadeNeutral()
//
//		On a mission change or if you join a server just before the match starts the script knows that all
//		the objectives are neutral.  It sets them neutral and then triggers this event.
//
//	eventObjectiveNew(%client, %id, %status, %type)
//
//		Just found about a new objective.  The event returns the client ID of the player that changed the
//		state of the objective, the ID of the objective, the status of the objective and the type of the
//		objective.  The client ID is 0 if it was the server that changed the status of the objective, for
//		instance returning an F&R flag after it was dropped and no one picked it up.
//
//	eventObjectiveUpdated(%client, %id, %status, %type)
//
//		An objective already known about has changed state in some way.  The event returns the client ID of
//		the player that took the objective, the ID of the objective, the new status of the objective and the
//		type of the objective.  The client ID is 0 if it was the server that changed the status of the
//		objective.
//
//
// New functions:
//
//	Objective::GetStatus(%id);
//
//		Returns the status of the objective that is associated with the ID.  Possible values returned
//		is dependant on the objective type.
//
//			C&H:
//				$Objective::Neutral
//					The match has just started, no one has control of the tower.
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Enemy
//					An enemy team has control of the tower.
//				$Objective::Friendly
//					Your team has control of the tower.
//
//			D&D:
//				$Objective::Neutral
//					The match has just started, the objective has not been destroyed.
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Destroyed
//					The objective has been destroyed.
//
//			F&R:
//				$Objective::Neutral
//					The match has just started, the flag is at its starting postition.
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Enemy
//					The flag is in the enemy base.
//				$Objective::Friendly
//					The flag is in your base.
//				$Objective::FriendlyCarry
//					Either you or a team mate is carrying the flag.
//				$Objective::EnemyCarry
//					An enemy is carrying the flag.
//				$Objective::Dropped
//					The flag has been dropped in the field
//
//	Objective::GetName(%id);
//
//		Get the name of the objective associated with the ID.
//
//	Objective::GetNameStripped(%id);
//
//		Get the name of the objective associated with the ID but strip off a "the" if it exists.  For
//		example if the name of the an objective was "the Central Citadel" this function would return
//		"Central Citadel".
//
// Objective::GetNameAbbreviated(%id)
//
//		Get the name of the objective associated with the ID but abbreviate the name.  This makes the
//		objective name a lot shorter in many cases.  For instance "Secondary Generator" is abbreviated to
//		"2nd Gen".  Take a look at the bottom of this script for a list of abbreviations.
//
//	Objective::Number();
//
//		Get the number of objectives we currently know about, this is the number loaded from disk plus
//		those we have found out about since.
//
//	Objective::GetId(%objectiveName);
//
//		Get the ID of an objective from its name.  Note that this has to be the entire name, "the Central
//		Citadel" in the example above.  Returns -1 if the objective doesn't exist.
//
//	Objective::GetType(%id);
//
//		Returns the type of this objective.  Values can be:
//			$Objective::CandH - Capture and Hold (tower)
//			$Objective::DandD - Defend and Destroy
//			$Objective::FandR - Find and Retrieve (flag)
//
//	Objective::GetClient(%id)
//
//		Get	the client involved with the last change of state.  In the case of someone carrying the flag this
//		would that players client number.  Otherwise it's whoever claimed, captured etc the objective.  If
//		there was no played involved in the change then the value is 0.  ie It was the server that made the
//		change.  This could happen in the case of returning a flag after a timeout for instance.
//
//	Objective::InsertTeamName(%msg)
//
//		In certain objective names we have to remember the team number involved, it is stored as a wild card.
//		For instance in a D&D type game: "%0 Main Generator".  This function searches for any wildcards and
//		inserts either "your" or "enemy" into it's place.  It even checks to see if word is at the start of
//		the sentance and so will capitalise the message.  In the example above: "Your Main Generator", if you
//		are on team 0.  This could've put the actual team name into the message but this makes the names very
//		long and for a user it is easier to work out who's objective it is.
//
//	Objective::PrintAll()
//
//		Useful for debuging, this function prints to the console the entire list of objectives currently
//		known about, with all their information.
//
//
// Changes in v2.0 beta 2:
//	+ Added support function Objective::GetNameAbbreviated(%id).  Abbreviations are stored in a table for
//	  efficiency.
//  + Had to add toggle key to distinguish between CTF and F&R flags, $ServerMissionType isn't always set :/.
//
// Changes in v2.0 beta:
//	+ Compelete rewrite.  Generalised the script so that it could track any objective.
//  + Beta version released to get bug reports as testing is difficult.
//  + Used the logic and messages from scripts.vol to derive the messages and parsing functions.
//
// Changes in v1.0.1:
//	= Fixed bug where Objective::GetNameStripped(%id) would return a null value if there was no "the" at
//	  the start of the name (don't rename variables ;).
//
// What happend to v1.0?
//	+ I first wrote the script to track C&H mission objectives but Presto pointed out it could be
//	  generalised to track any objective.  Thanks for the idea Presto :)
//
//
// To do:
//  + Multi-team names?
//  + Multi-team CTF?
//
//
// -- Preferences --
//
// If $ServerMissionType has not been set on a server the script will not know the difference between CTF and
// F&R flags.  You can set which the script should be tracking and it will fall back on this if the variable
// hasn't been set on the server.
$CrunchyPref::FlagTogKey = "alt a";
//

//
// -- Header --
//

Include("Presto\\Event.cs");
Include("Crunchy\\Events.cs");
Include("Presto\\TeamTrak.cs");
Include("Presto\\Match.cs");

Event::Attach(eventChangeMission, "Objective::Save();Objective::Reset();$MidGame = false;");
Event::Attach(eventConnectionAccepted, "Objective::Reset();$MidGame = true;");
Event::Attach(eventClientMessage, Objective::Parse);
Event::Attach(eventMissionInfo, Objective::Load);

//
// -- Begin code --
//

// Toggle to set mode if $ServerMissionType has not been set.
bindKey(play, $CrunchyPref::FlagTogKey, "Objective::ToggleFlagType();");
$Objective::ToggleFlag = ctf; // Defaults to CTF flags.
function Objective::ToggleFlagType()
{
	if($Objective::ToggleFlag == ctf)
	{
		$Objective::ToggleFlag = fandr;
		%msg = "F&R";
	}
	else
	{
		$Objective::ToggleFlag = ctf;
		%msg = "CTF";
	}

	remoteBP(2048, "<f0>	Tracking <f2>"@%msg@"<f0> flags",5);
}

// Function to parse the incoming messages.  Compares the message to a list we have created and triggers
// an action function if we have a match.
function Objective::Parse(%client, %msg)
{
	// Loop through all the messages.
	for(%i = 0; %i < $ObjMsg::Num; %i++)
	{
		// On a match call the associated action function.
		if(Match::ParamString(%msg, $ObjMsg::[%i]))
		{
			%p = Match::Result(p);
			%o = Match::Result(o);
			%t = Match::Result(t);
			%action = $ObjMsg::[%i, action];
			eval("Objective::Action_"@%action@"(%p, %o, %t);");

			return true;
		}
	}

	// Watch for a match start and make the objectives neutral in that case.
	if(Match::String(%msg, "Match starts in * seconds.") || %msg=="Match started.")
	{
		if($Objective::Loaded && $MidGame)
			Objective::MakeNeutral();

		$MidGame = false;
	}

	return true;
}

// Update value of an objective.
function Objective::Set(%client, %o, %status, %type)
{
	%id = Objective::GetId(%o);
	if(%id == -1)
	{
		// Objective doesn't exist so create a new one.
		%id = $ObjData::Num;
		$ObjData::Num++;
		%new = true;
		$Objective::New = true;
		$ObjData::[%id, abrv] = Objective::Abbreviate(%o);
	}
	else
		%new = false;

	// Update the objective.
	$ObjData::[%id, client] = %client;
	$ObjData::[%id, name] = %o;
	$ObjData::[%id, status] = %status;
	$ObjData::[%id, type] = %type;

	// Interpret the status.
	%status = eval("Objective::Interpret_"@%type@"($ObjData::[%id, status]);");
	// Triggers.
	if(%new)
		Event::Trigger(eventObjectiveNew, %client, %id, %status, %type);
	else
		Event::Trigger(eventObjectiveUpdated, %client, %id, %status, %type);
}

// Load the objective list off disk if available.
function Objective::Load(%server, %missionName, %missionType)
{
	%file = %missionName@".cs";
	if(isFile("config\\"@%file))
	{
		exec(%file);
		$Objective::Loaded = true;
		Event::Trigger(eventObjectiveLoaded, Objective::Number());
		Objective::MakeAbbreviations();
		if(!$MidGame) Objective::MakeNeutral();
	}
	else if(isFile("config\\Missions\\"@%file))
	{
		exec("Missions\\"@%file);
		$Objective::Loaded = true;
		Event::Trigger(eventObjectiveLoaded, Objective::Number());
		Objective::MakeAbbreviations();
		if(!$MidGame) Objective::MakeNeutral();
	}
}

// Save out to disk the objective list we created.
function Objective::Save()
{
	if(!$Objective::New) return;

	%file = "config\\"@$ServerMission@".cs";
	export("$ObjData::Num", %file, false);
	export("$ObjData::*_name", %file, true);
	export("$ObjData::*_type", %file, true);
}

// If we are starting a mission where we know about some of the objectives we know they are neutral.
function Objective::MakeNeutral()
{
	for(%id = 0; %id < Objective::Number(); %id++)
		$ObjData::[%id, status] = $Objective::Neutral;

	Event::Trigger(eventObjectiveMadeNeutral);
}

// Make up the abbreviations once so we don't have to do this expensive operation all the time.
function Objective::MakeAbbreviations()
{
	for(%id = 0; %id < Objective::Number(); %id++)
		$ObjData::[%id, abrv] = Objective::Abbreviate(Objective::GetNameStripped(%id));
}

// Get the ID from the name of an objective.
function Objective::GetId(%o)
{
	// Not efficient but we aren't dealing with large numbers of objectives.
	for(%id = 0; %id < Objective::Number(); %id++)
		if(%o == Objective::GetName(%id)) return %id;

	return -1; // Objective not found.
}

// Inefficient, only use if you HAVE too :)
// This is used once, and then in a rare case.
function Objective::GetCarry(%client)
{
	for(%id = 0; %id < Objective::Number(); %id++)
	{
		if(Objective::GetStatus(%id) == $Objective::FriendlyCarry
		|| Objective::GetStatus(%id) == $Objective::EnemyCarry)
		{
			if(Objective::GetClient(%id) == %client)
				return %id;
		}
	}

	return "";
}

// Work out client from the message.
function Objective::GetClientFromMsg(%p)
{
	if(%p == "You")
		return getManagerId();
	else
		return getClientByName(%p);
}

// Number of those we know about up to now.
function Objective::Number()
{
	return $ObjData::Num;
}

// Get the name of an objective.
function Objective::GetName(%id)
{
	return $ObjData::[%id, name];
}

// Strip the "the" from a name.
function Objective::GetNameStripped(%id)
{
	%o = Objective::GetName(%id);
	if(Match::ParamString(%o, "the %o"))
		return Match::Result(o);
	else
		return %o;
}

// Get the abbreviated version of the objective name.
function Objective::GetNameAbbreviated(%id)
{
	return $ObjData::[%id, abrv];
}

// Get the current status of an objective in terms of the $Objective::* variables.
function Objective::GetStatus(%id)
{
	%type = Objective::GetType(%id);
	%status = eval("Objective::Interpret_"@%type@"($ObjData::[%id, status]);");
}

// Get the type of the objective.
function Objective::GetType(%id)
{
	return $ObjData::[%id, type];
}

// Get the client number involved in the last status change.
function Objective::GetClient(%id)
{
	return $ObjData::[%id, client];
}

// Clear the list between missions.
function Objective::Reset()
{
	deleteVariables("$ObjData::*");
	$ObjData::Num = 0;
	$Objective::Loaded = false;
	$Objective::New = false;

	Event::Trigger(eventObjectiveReset);
}

// Useful for debuging, prints the name and status of an objective.
function Objective::Print(%id)
{
	echo("Player: ", Client::getName(Objective::GetClient(%id)));
	echo("Name: ", Objective::GetName(%id));
	echo("Status: ", Objective::GetStatus(%id));
	echo("Type: ", Objective::GetType(%id));
}

// Print the names and status of all the objectives we know about.
function Objective::PrintAll()
{
	echo("Friendly Team: ", Team::GetName(Team::Friendly()));
	echo("Enemy Team: ", Team::GetName(Team::Enemy()));
	for(%id = 0; %id < Objective::Number(); %id++)
		Objective::Print(%id);
}

// Setup message database.
$ObjMsg::Num = 0;
function Objective::Message(%msg, %action, %type)
{
	%i = $ObjMsg::Num;
	$ObjMsg::[%i] = %msg;
	$ObjMsg::[%i, action] = %action;
	$ObjMsg::Num++;
}

// Setup abreviations.
function Objective::AbbreviationReset()
{
	$ObjAbrv::Num = 0;
}
function Objective::Abbreviation(%name, %len, %abrv)
{
	%i = $ObjAbrv::Num;
	$ObjAbrv::[%i] = %name;
	$ObjAbrv::[%i, len] = %len;
	$ObjAbrv::[%i, abrv] = %abrv;
	$ObjAbrv::Num++;
}

// Abbreviate objective names.
function Objective::Abbreviate(%msg)
{
	%abrv = true;
	while(%abrv)
	{
		%abrv = false;
		for(%i = 0; %i < $ObjAbrv::Num; %i++)
		{
			if((%pos = String::findSubStr(%msg, $ObjAbrv::[%i])) != -1)
			{
				%msg = String::getSubStr(%msg, 0, %pos) @
					   $ObjAbrv::[%i, abrv] @
					   String::getSubStr(%msg, %pos + $ObjAbrv::[%i, len], 256);
				%abrv = true;
			}
		}
	}

	return %msg;
}

// Work out the team the message was refering to.
// No longer used, but could be useful :)
function Objective::GetTeamDet(%d)
{
	if(%d == "an")
		return Team::Friendly();
	else
		return Team::Enemy();
}

// Where the name of an objective is dependant on the team, the team number is stored as "%0" etc.  When we
// display it we want to insert "Your" or "Enemy" in the play of these wildcards.
function Objective::InsertTeamName(%msg)
{
	while((%pos = String::findSubStr(%msg, "%")) != -1)
	{
		%team = String::getSubStr(%msg, %pos+1, 1);  // Assumes not more than 10 teams :)
		if(%team == Team::Friendly())
		{
			if(%pos == 0)
				%teamName = "Your";
			else
				%teamName = "your";
		}
		else
		{
			if(%pos == 0)
				%teamName = "Enemy";
			else
				%teamName = "enemy";
		}
		%msg = String::getSubStr(%msg, 0, %pos) @ %teamName @ String::getSubStr(%msg, %pos+2, 256);
	}

	return %msg;
}

//
// The plugins
// ===========
//
// This is the cool bit :).  Here we add the messages that we are going to parse.  Each message has an
// associated action function.  The message provides the syntax and the action function gives the semantics.
// In the action function we set the new status of the objective and do any processing required.  These
// functions don't have to be stored in this file.  As long as a script containing more plugins is executed
// alongside this script they will get used.
//
//
// Status values
// -------------
//
// The status of the objective can be one of the following things, the actual meaning depends on the
// objective in question.  All objectives can be unknown or neutral however.  The status is returned by the
// Objective::GetStatus(%id).  Note that the internal representation of the state can be different, these
// values are the ones handed to the end user.  For instance store something team dependant using team
// numbers and return the status $Objective::Enemy if the team number stored isn't the same as the players
// team number.
//
$Objective::Enemy			= enemy;
$Objective::Friendly		= friendly;
$Objective::Destroyed		= destroyed;
$Objective::Neutral			= neutral;
$Objective::FriendlyCarry	= carryFriendly;
$Objective::EnemyCarry		= carryEnemy;
$Objective::Dropped			= dropped;
$Objective::Unknown			= unknown;
//
//
// Objective types
// ---------------
//
// The types currently supported.  This is returned by the Objective::GetType(%id) function.
//
$Objective::CandH			= candh;
$Objective::DandD			= dandd;
$Objective::FandR			= fandr;
//
//
// Action functions
// ----------------
//
// Each message has an action function associated with it.  The function is passed three variables, see the
// messages section.  Messages could share the same action function.  In these functions you can do whatever
// you need to, this script just provides a way to keep track of the objectives.  For instance you could
// trigger extra events.
//
// The minimum the action function should provide is running the following function with appropriate values:
//		Objective::Set(%client, %o, %status, %type);
// 			This function either creates or updates the objective.
// Note that this function doesn't have to be called straight away, it is sometimes nessary to delay the call
// until we have more information provided by another message.  Use two or more action functions in tandom
// for this.
//
// Objective::Action_"action"(%p, %o, %t);
//		The action function is always named Objective::Action_ followed by the name of the action.  See the
//		message section.
//
// claimed: an objective has just been claimed.
function Objective::Action_claimed(%p, %o, %t)
{
	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);
	Objective::Set(%client, %o, %team, candh);

	$ObjData::Team = %team;
}
//
// captured: an objective has been captured.
function Objective::Action_captured(%p, %o, %t)
{
	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);
	if($ServerMissionType != "Multiple Team") Team::SetName(Team::Enemy(%client), %t);
	Objective::Set(%client, %o, %team, candh);

	$ObjData::Team = %team;
}
//
// destroy: Just destroyed an objective.
function Objective::Action_destroy(%p, %o, %t)
{
	%team = $ObjData::Team;
	Team::SetName(%team, %t);
	Objective::Set($ObjData::Client, "%"@%team@" "@%o, $Objective::Destroyed, dandd);
}
//
// teamName: we just found out about a team name.
function Objective::Action_teamName(%p, %o, %t)
{
	if(Match::ParamString(%t,"%t team"))
		%t = Match::Result(t);
	Team::SetName($ObjData::Team, %t);
}
//
// playerEnemy: get an enemy player name we need for later on.
function Objective::Action_playerEnemy(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	$ObjData::Team = Team::Enemy($ObjData::Client);
}
//
// playerFriendly: get a friendly player name we need for later on.
function Objective::Action_playerFriendly(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	$ObjData::Team = Client::getTeam($ObjData::Client);
}
//
// carry: player has picked up a flag and is carrying it.
function Objective::Action_carry(%p, %o, %t)
{
	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);
	$ObjData::Team = %team;

	if(Match::ParamString(%o, "%o from the %t team"))
	{
		%o = Match::Result(o);
		%t = Match::Result(t);
		Team::SetName(Team::Enemy(%client), %t);
	}

	if(%team == Team::Friendly())
		%status = $Objective::FriendlyCarry;
	else
		%status = $Objective::EnemyCarry;

	Objective::Set(%client, %o, %status, fandr);
}
//
// dropped: flag has been dropped in the field.
function Objective::Action_dropped(%p, %o, %t)
{
	// Not implemented yet (this stops spurious objectives being created):
	if($ServerMissionType == "Capture the Flag" || $ServerMissionType == "Multiple Team") return;
	if($Objective::ToggleFlag == ctf && $ServerMissionType != "Find and Retrieve") return;

	%client = Objective::GetClientFromMsg(%p);
	Objective::Set(%client, %o, $Objective::Dropped, fandr);
}
//
// neutral: flag was returned to neutral stand.
function Objective::Action_neutral(%p, %o, %t)
{
	// One of those cases where two messages bump into each other.
	if(Match::ParamString(%o, "%p left the mission area while carrying %o!  It"))
	{
		%o = Match::Result(o);
		%client = Objective::GetClientFromMsg(Match::Result(p));
	}
	else
		%client = 0;

	Objective::Set(%client, %o, $Objective::Neutral, fandr);
}
//
// conveyed: flag has been conveyed to base.
function Objective::Action_conveyed(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	$ObjData::Team = Client::getTeam($ObjData::Client);
}
//
// returned: flag was returned to base.
function Objective::Action_returned(%p, %o, %t)
{
	// Not implemented yet (this stops spurious objectives being created)
	if($ServerMissionType == "Capture the Flag" || $ServerMissionType == "Multiple Team") return;

	if(%t == "Your")
		$ObjData::Team = Team::Friendly();
	else
	{
		$ObjData::Team = Team::Enemy();
		if($Objective::PlayerLeft)
		{
			if(Match::ParamString(%t, "The %t"))
			{
				%t = Match::Result(t);
				Team::SetName($ObjData::Team, %t);
			}
		}
	}

	if($Objective::PlayerLeft)
	{
		Objective::Set($ObjData::Client, $ObjData::Name, $ObjData::Team, fandr);
		$Objective::PlayerLeft = false;
	}
	else
		$ObjData::Client = 0;
}
//
// holds: a team holds one of the flags
function Objective::Action_holds(%p, %o, %t)
{
	Objective::Set($ObjData::Client, %o, $ObjData::Team, fandr);
	if(%t != "Your")
		Team::SetName($ObjData::Team, %t); // Assuming two teams.
}
//
// playerLeft: a player left the mission area whilst carrying a flag.
function Objective::Action_playerLeft(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	// The message doesn't tell us the objective so we have to lookup what the player was carrying.
	$ObjData::Name = Objective::GetName(Objective::GetCarry($ObjData::Client));
	$Objective::PlayerLeft = true;
}
//
// cap: a flag was just captured.
function Objective::Action_cap(%p, %o, %t)
{
	// CTF still to do :)
}
//
//
// Interpreting status
// -------------------
//
// We don't necessarily pass the stored internal state to the user.  It probably needs interpretation, for
// instance if we store team numbers.  This function is called by Objective::GetStatus(%id).
//
// Objective::Interpret_"type"(%status);
//		The interpret function is always named Objective::Interpret_ followed by the type of the objective.
//
function Objective::Interpret_candh(%status)
{
	if(%status == Team::Friendly())
		return $Objective::Friendly;
	else if(%status == "")
		return $Objective::Unknown;
	else if(%status == $Objective::Neutral)
		return %status;
	else
		return $Objective::Enemy;  // Works with multiple teams.  Can't tell you exact enemy team though.
}
//
function Objective::Interpret_dandd(%status)
{
	if(%status == "")
		return $Objective::Unknown;
	else
		return %status;
}
//
function Objective::Interpret_fandr(%status)
{
	if(%status == Team::Friendly())
		return $Objective::Friendly;
	else if(%status == Team::Enemy())
		return $Objective::Enemy;
	else if(%status == "")
		return $Objective::Unknown;
	else
		return %status;
}
//
function Objective::Interpret_ctf(%status)
{
	if(%status == "")
		return $Objective::Unknown;
	else
		return %status;
}
//
//
// Objective messages
// ------------------
//
// Here we add the actual messages to parse.
//
// Objective::Message(%msg, %action);
//		%msg: Text message.
//		%action: Function to be called if we have a match.
//
// The values that are passed to the action function:
//		%p: player name
//		%o: objective name
//		%t: a team name
//
//		%z: punctuation etc, _not_ passed to the functions.
//
// C&H
Objective::Message("%p claimed %o for the %t team!", claimed);
Objective::Message("%p captured %o from the %t team!", captured);
Objective::Message("The %t has taken %z objective.", teamName); // %z is "an" or "your".
//Objective::Message("Your team has taken an objective.", teamName);  // Can't learn anything.
//
// D&D
Objective::Message("%p destroyed a friendly objective%z", playerFriendly); // %z is "." or "!".
Objective::Message("%p destroyed an objective!", playerEnemy);
Objective::Message("%t objective %o destroyed.", destroy);
//
// F&R (what a nightmare :)
Objective::Message("%p took %o.", carry);
//Objective::Message("%p took %o from the %t team.", carry); // Covered by above msg.
//Objective::Message("Your team has %o.", carry); // Can't learn anything.
//Objective::Message("Your team lost %o.", carry); // Can't learn anything.
Objective::Message("The %t team has taken %o.", teamName);
Objective::Message("%p dropped %o!", dropped);
Objective::Message("%o was returned to its initial position.", neutral);
Objective::Message("%t flag was returned to base.", returned);
Objective::Message("%t team holds %o.", holds);
Objective::Message("%p conveyed %o to base.", conveyed);
Objective::Message("%p left the mission area while carrying the %t flag!", playerLeft);
//Objective::Message("Flag not in mission area.", ?); // Doesn't effect status.
//
// CTF - to do?
//Objective::Message("%p took %o! ", carry);
//Objective::Message("%p dropped %o!", dropped); // Same as F&R dropped.
//Objective::Message("Your flag was dropped in the field.", dropped);  // Can't learn anything.
//Objective::Message("%o was dropped in the field.", dropped);  // Can't learn anything.
//Objective::Message("%p returned %o!", returned);
//Objective::Message("%o was returned to base.", returned); // Same as F&R returned.
//Objective::Message("%p captured %o!", cap);
//
//
// Objective name abbreviations
// ----------------------------
//
//	Objective::AbbreviationReset();
//		Start making a list of abbreviations.
//
//	Objective::Abbreviation(%word, %wordLen, %abrv);
//		%word: Word to abbreviate.
//		%wordLen: Length of word to abbreviate.
//		%abrv: Abbreviation to use in place of accurances of the word.
//
Objective::AbbreviationReset();
Objective::Abbreviation("Generator",9,"Gen");
Objective::Abbreviation("Secondary",9,"2nd");
Objective::Abbreviation("#",1,"");
Objective::Abbreviation("Northwest", 9, "NW");
Objective::Abbreviation("Southwest", 9, "SW");
Objective::Abbreviation("Northeast", 9, "NE");
Objective::Abbreviation("Southeast", 9, "SE");
//Objective::Abbreviation("Repository", 10, "Reposit.");
Objective::Abbreviation("Central Pyramid", 15, "Pyramid");
Objective::Abbreviation("Abandoned Dropship", 18, "Dropship");
Objective::Abbreviation("Pulse Sensor", 12, "Sensor");
Objective::Abbreviation("Plasma Turret", 13, "Plasma");
Objective::Abbreviation("Command", 7, "Comm");
//