// -- Objective HUD -- v2.4 ---------------------------------------------------------------------------------
// Written by |HH|Crunchy (Jonathan Slark)
// Web page: http://www.crunchy.oaktree.co.uk/scripting/
//
// *** Requires the Presto Pack v0.93 or later ***  http://www.planetstarsiege.com/presto/
// *** Requires Zear's NewOpts v0.95 or later *** http://www.cetisp.com/~thammock/scripts/
// *** Requires my ObjectiveTrak script, a new tool for scripters! v2.0 beta 7 or later ***
// *** Requires my Events script ***
// To install, put the scripts and pictures into a directory named "Crunchy" in your "Tribes/config"
// directory.  Include the ObjectiveHUD.cs in your autoexec.cs.  Now read the notes below and check out the 
// options menu.
// example autoexec.cs:
//	exec("Presto\\Install.cs");
//	Include("Crunchy\\ObjectiveHUD.cs");
//
// The Objective HUD displays the names and status of each of the objectives in a mission. The HUD uses my
// ObjectiveTrak script, the script won't know about objectives until they are captured, dropped etc.
// However the script saves the information to disk for use in the future.  Play your favourite mission and
// once it knows about all the objectives it will display them all everytime you play the level.  The script
// currently works with CTF, C&H, D&D and F&R type objectives.  C&H objectives are tracked correctly in
// multiple team missions.  There is a file pack available that has information about all the base missions.  
// If you download and install this then the HUD will know about the objectives without having to learn about
// them.
//
// CTF
// Your flag starts off green with grey background, it is safe in your base.  The enemy flag is shown as red
// with grey background when it is safe in the enemy base.  When someone grabs a flag they will have a yellow
// background with a green or red square on it.  Green means friendly carrier, red is enemy carrier.  A 
// yellow background and black square is a dropped flag.  A yellow sqaure on a grey background indicates 
// unknown status.  Hard to visualise but it's obvious when you are playing :).  See options menu / help.
//
// C&H
// These objectives have a switch symbol: red means enemy controled, green means your team has control and
// black means neutral (ie start of the game).  If you join mid-game then the script won't know the status of
// the objectives, they will be yellow.
//
// D&D
// These objectives appear as generator symbols.  Red means destroyed, green means safe.  Again yellow means
// unknown.  Both the enemy and friendly objectives start out green and will turn red once destroyed. If you
// set the abbreviate preference to true a green player symbol next to the status symbol means it is one of
// your objectives, defend it!  If the player symbol is red, this objective should be destroyed!
//
// F&R
// The flags start off black: neutral.  When someone picks up a flag they will have a yellow background with
// a green or red square on it.  Green means friendly carrier, red is enemy carrier.  Grey background with
// red or green square on the flag means the enemy has the flag at their base or you do respectively.  A
// yellow sqaure on a grey background indicates unknown status or dropped flag.
//
//
// Changes in v2.4:
//  + HUDMover friendly (you CAN teach an old dog new tricks :).
//  + Rewritten update routine so that only the lines that need to be updated are updated.  This should help 
//	  with CPU lag, especially with lots of objectives.
//	+ If ppl have control characters in their name this could mess up the display, these are now replaced 
//	  with valid ones.  Eg: "<oep>Nachoking" becomes "[oep]Nachoking".  The name is also stripped of any 
//	  leading or trailing spaces by using String::trim from NewOpts.
//  = Various bug fixes and changes, esp regarding events.
//
// Changes in v2.3.2:
//	+ Added eventObjectiveHUDResize (used by TeamScoreHUD).
//  = When observing, CTF flags are displayed with the team name (if known) instead of "Enemy flag" (change 
//	  in ObjectiveTrak).
//  = Changed dropped flag icon to be clearer and more consistent with the other icons.
//
// Changes in v2.3.1:
//	= Fixed bug where options screen would mess up if used in a resolution other than 640x480.
//
// Changes in v2.3:
//	+ Added options page for Zear's NewOpts!  Thanks to Zear for NewOpts :)
//  + Saves and loads last preferences, so you set 'em once and they stick :)
//  + Options can be set on the fly and the HUD updates properly.
//  + Solved the "messed up HUD" bug by puting each line into a SimGui::Control so that they are clipped 
//	  properly, Thanks to Cowboy for the idea!  Thanks to Zear for his gui docs!
//	= Moved location of the bmps to their own directory, if you installed a previous version you can remove 
//	  the bmps in the Tribes\Crunchy dir, they are now in the bmp dir.
//
// Changes in v2.2.1:
//  = Client message parser didn't check to see if the message was from the server. D'oh :)
//    (this is actually a change in ObjectiveTrak but it's important it was fixed)
//
// Changes in v2.2:
//	+ Added optional CTF flag support!
//  = Fixed minor bug in blinking code (the code to make the names blink, didn't mean I was fed up with it :)
//  = Stopped two consecutive updates to same objective making the blinking schedules clash.
//
// Changes in v2.1.1:
//	= Fixed bug where HUD wouldn't be updated if no objectives loaded.
//
// Changes in v2.1:
//  + Overcome 8 bitmap limit by having a seperate object per line.  Thanks Cowboy for the suggestion!
//  + Added preference for user to decide if the HUD should always come on, or only when toggled with a key,
//	  or when there are objectives on a mission.
//  + Added option to abbreviate objective names and thus make the HUD _a lot_ smaller.
//  + Objectives blink for a few seconds when they have been updated.
//  + Preferences can be put in a seperate file and they will override preferences in this file.
//  = Some changes on what functions were attached to events and where.
//
// Changes in v2.0.1:
//	+ HUD comes on automatically at start of any mission type.  The HUD is useful for most missions.
//
// What no v1.0?
//  - Version 1 worked with ObjectiveTrak v1.0 which only tracked C&H objectives.  I generalised the tracking
//	  script to track any objectives.
//
//
// -- Preferences --
//
// Preferences are now set using a NewOpts page!  Start Tribes then go to Options, Scripts and then select 
// ObjectiveHUD from the pull down list.  Now you can change the options, get some help and then the options 
// are saved so you only need to set all this once.
//

//
// -- Header --
//

if ($Presto::version < 0.93)
	echo("Objective HUD requires Presto Pack v0.93 or later.");
else
{

function ObjectiveHUD::SetDefaults()
{
	$CrunchyPref::ObjectiveHUDDisplay = true;
	$CrunchyPref::ObjectiveAbbreviate = true;
	$CrunchyPref::ObjectiveCTF = true;
	$CrunchyPref::ObjectiveHUDPos = "100% 50% 126 14";
}
if(isFile("config\\CrunchyPrefs.cs"))
{
	// Load last prefs off disk
	Include("CrunchyPrefs.cs");  // We only want this run once, a script might want to override.

	// In the case the defaults don't contain one for Objective HUD
	if($CrunchyPref::ObjectiveHUDPos == "")
		ObjectiveHUD::SetDefaults();
}
else
	ObjectiveHUD::SetDefaults();

Include("Presto\\Event.cs");
Include("Crunchy\\Events.cs");
Include("Presto\\HUD.cs");
Include("Presto\\TeamTrak.cs");
Include("Crunchy\\ObjectiveTrak.cs"); // This does the business, a new tool for scripters!

Event::Attach(eventClientChangeTeam, ObjectiveHUD::onChangeTeam);
Event::Attach(eventObjectiveReset, "ObjectiveHUD::onReset();");
Event::Attach(eventObjectiveNew, ObjectiveHUD::onNew);
Event::Attach(eventObjectiveUpdated, ObjectiveHUD::onUpdate);
Event::Attach(eventObjectiveLoaded, "ObjectiveHUD::onLoaded();");

$ObjectiveHUD::CharsNotAllowed	= "<>";
$ObjectiveHUD::CharsReplace 	= "[]";

// Hand drawn and hand anti-aliased BMPs (you try and make pictures this small look cool ;)
// If ppl are using massive resolutions I may need to make these clearer?
// |HH|Rico assures me they are ok :)

$ObjectiveHUD::dirBMP = "Crunchy\\bmp\\";

// C&H
$ObjectiveHUD::[$Objective::Friendly,$Objective::CandH]=$ObjectiveHUD::dirBMP@"small_tower_friendly";
$ObjectiveHUD::[$Objective::Enemy,$Objective::CandH]=$ObjectiveHUD::dirBMP@"small_tower_enemy";
$ObjectiveHUD::[$Objective::Unknown,$Objective::CandH]=$ObjectiveHUD::dirBMP@"small_tower_unknown";
$ObjectiveHUD::[$Objective::Neutral,$Objective::CandH]=$ObjectiveHUD::dirBMP@"small_tower_neutral";

// D&D
$ObjectiveHUD::[$Objective::Neutral,$Objective::DandD]=$ObjectiveHUD::dirBMP@"small_objective_safe";
$ObjectiveHUD::[$Objective::Destroyed,$Objective::DandD]=$ObjectiveHUD::dirBMP@"small_objective_destroyed";
$ObjectiveHUD::[$Objective::Unknown,$Objective::DandD]=$ObjectiveHUD::dirBMP@"small_objective_unknown";

// CTF
$ObjectiveHUD::[$Objective::Friendly,$Objective::CTF]=$ObjectiveHUD::dirBMP@"small_flag_friendly";
$ObjectiveHUD::[$Objective::Enemy,$Objective::CTF]=$ObjectiveHUD::dirBMP@"small_flag_enemy";
$ObjectiveHUD::[$Objective::FriendlyCarry,$Objective::CTF]=$ObjectiveHUD::dirBMP@"small_flag_friendlyCarry";
$ObjectiveHUD::[$Objective::EnemyCarry,$Objective::CTF]=$ObjectiveHUD::dirBMP@"small_flag_enemyCarry";
$ObjectiveHUD::[$Objective::Dropped,$Objective::CTF]=$ObjectiveHUD::dirBMP@"small_flag_dropped";
$ObjectiveHUD::[$Objective::Unknown,$Objective::CTF]=$ObjectiveHUD::dirBMP@"small_flag_unknown";

// F&R
$ObjectiveHUD::[$Objective::Friendly,$Objective::FandR]=$ObjectiveHUD::dirBMP@"small_flag_friendly";
$ObjectiveHUD::[$Objective::Enemy,$Objective::FandR]=$ObjectiveHUD::dirBMP@"small_flag_enemy";
$ObjectiveHUD::[$Objective::FriendlyCarry,$Objective::FandR]=
	$ObjectiveHUD::dirBMP@"small_flag_friendlyCarry";
$ObjectiveHUD::[$Objective::EnemyCarry,$Objective::FandR]=$ObjectiveHUD::dirBMP@"small_flag_enemyCarry";
$ObjectiveHUD::[$Objective::Dropped,$Objective::FandR]=$ObjectiveHUD::dirBMP@"small_flag_dropped";
$ObjectiveHUD::[$Objective::Unknown,$Objective::FandR]=$ObjectiveHUD::dirBMP@"small_flag_unknown";
$ObjectiveHUD::[$Objective::Neutral,$Objective::FandR]=$ObjectiveHUD::dirBMP@"small_flag_neutral";

function ObjectiveHUD::AddBanner()
{
	Presto::AddScriptBanner(ObjectiveHUD,
		" <f2>Objective HUD <jr><f0>version 2.4 <jl>\n" @
		" \n" @
		" <f0>Displays mission objectives.\n" @
		" <f0>Now uses <f2>NewOpts<f0>! You can\n" @
		" <f0>set your preferences in the\n" @
		" <f0>Options menu and move the\n" @
		" <f0>HUD with HUDMover.\n" @
		" \n" @
		" <f0>Written by: <f1>|HH|Crunchy\n");
}
ObjectiveHUD::AddBanner();

//
// -- Begin code --
//

// If the text displayed in a FearGuiFormattedText object has characters such as < they are taken to mean 
// control characters and so mess up the display.  This function is used to replace these chars with a
// valid char.
function FilterFormattedText(%text, %charNotAllowed, %charReplace)
{
	%text = String::trim(%text);
	for(%i = 0; %i < String::len(%text); %i++)
	{
		%char = String::getSubStr(%text, %i, 1);
		if((%pos = String::findSubStr(%charNotAllowed, %char)) != -1)
			%text = String::getSubStr(%text,0,%i)@
					String::getSubStr(%charReplace,%pos,1)@
					String::getSubStr(%text, %i+1, 255);
	}

	return %text;
}

// Update one line of the HUD.
function ObjectiveHUD::UpdateLine(%id)
{
	%type = Objective::GetType(%id);
	// DeMorgan's law used to simplify:
	// !(%type == $Objective::CTF && !$CrunchyPref::ObjectiveCTF)
	if(%type != $Objective::CTF || $CrunchyPref::ObjectiveCTF)
	{
		%status = Objective::GetStatus(%id);
		%bmp = $ObjectiveHUD::[%status, %type];

		if($ObjectiveHUD::BlinkOn[%id])
			%colour = "<f2>";
		else
			%colour = "<f1>";

		if((%score = Objective::GetScore(%id)) == 0) %score = Objective::GetDeltaScore(%id);

		if(%type == $Objective::CTF)
		{
			// If a flag is carried show the player's name rather than the flag's name.
			if(%status == $Objective::FriendlyCarry || %status == $Objective::EnemyCarry)
			{
				%o = Client::GetName(Objective::GetClient(%id));
 				%o = FilterFormattedText(%o,$ObjectiveHUD::CharsNotAllowed,$ObjectiveHUD::CharsReplace);
			}
			else
				%o = Objective::InsertTeamName(Objective::GetName(%id));
		}
		else if($CrunchyPref::ObjectiveAbbreviate)
		{
			%o = Objective::GetNameAbbreviated(%id);
			if(%type == $Objective::DandD) %o = Objective::InsertTeamSymbol(%o);
		}
		else
		{
			%o = Objective::GetNameStripped(%id);
			if(%type == $Objective::DandD) %o = Objective::InsertTeamName(%o);
		}

		Control::SetValue(Object::GetName($ObjectiveHUD::Line[$ObjectiveHUD::LookupLine[%id]]),
						  %colour@"<B0,4:"@%bmp@".bmp><L4>"@%o);
	}
}

// Update the HUD using ObjectiveTrak.
function ObjectiveHUD::UpdateAll()
{
	%num = Objective::Number();

	if(%num == 0 || (!$CrunchyPref::ObjectiveCTF && ObjectiveHUD::NonCTFNumber() == 0))
		Control::SetValue(Object::GetName($ObjectiveHUD::Line[0]), "<jl><f1>No objectives");
	else
	{
		for(%id = 0; %id < %num; %id++)
			ObjectiveHUD::UpdateLine(%id);
	}
}

// If we find out about new objectives resize the HUD accordingly.
function ObjectiveHUD::Resize()
{
	if($CrunchyPref::ObjectiveCTF)
		%numLines = Objective::Number();
	else
		%numLines = ObjectiveHUD::NonCTFNumber();
	if(%numLines < 1) %numLines = 1;
	%lines = HUD::GetGuiObjectCount(ObjectiveHUD); // Number of existing lines.
	%perLine = getWord($CrunchyPref::ObjectiveHUDPos, 3); // Height per line.
	%width = HUD::GetCoordValue(ObjectiveHUD, 2);

	// If we don't already have objects for the lines, make em.
	for(%i = %lines; %i < %numLines; %i++)
	{
		$ObjectiveHUD::Frame[%i] =
			HUD::AddObject(ObjectiveHUD, SimGui::Control, 4, %i * %perLine - 3, %width - 8, %perLine+2);
		$ObjectiveHUD::Line[%i] =
			newobject("$ObjectiveHUD::_Line"@%i, FearGuiFormattedText, 0, 0, %width - 8, %perLine+2);
		addToSet($ObjectiveHUD::Frame[%i], Object::GetName($ObjectiveHUD::Line[%i]));
	}

	// Resize the HUD.
	HUD::SetCoord(ObjectiveHUD, height, %numLines * %perLine);
	ObjectiveHUD::AssignRows();
	ObjectiveHUD::UpdateAll();

	// Let other scripts know the HUD has been resized.
	Event::Trigger(eventObjectiveHUDResize);
}

// Get number of non-CTF objectives.
function ObjectiveHUD::NonCTFNumber()
{
	%num = 0;
	for(%id = 0; %id < Objective::Number(); %id++)
	{
		if(Objective::GetType(%id) != $Objective::CTF)
			%num++;
	}
	return %num;
}

// If we change teams HUD needs updating.
function ObjectiveHUD::onChangeTeam(%client, %team)
{
	if(%client != getManagerID()) return;

	ObjectiveHUD::UpdateAll();
}

// If objectives were loaded or new ones found about resize the HUD.
// Also if pref onLoaded is set display HUD.
function ObjectiveHUD::onLoaded()
{
	if(($CrunchyPref::ObjectiveCTF || ObjectiveHUD::NonCTFNumber())
	&& $CrunchyPref::ObjectiveHUDDisplay == onLoaded)
		HUD::Display(ObjectiveHUD, true);

	ObjectiveHUD::Resize();
}

// Work out which lines the objective id's are associated with.
// If we aren't displaying CTF flags this isn't linear, plus we could change the order if we want.
function ObjectiveHUD::AssignRows()
{
	%row=0;
	for(%id = 0; %id < Objective::Number(); %id++)
	{
		if(Objective::GetType(%id) != $Objective::CTF || $CrunchyPref::ObjectiveCTF)
		{
			$ObjectiveHUD::LookupLine[%id] = %row;
			%row++;
		}
	}
}

// When an objective has been found out about, update HUD plus blink the name.
function ObjectiveHUD::onNew(%client, %id, %status, %type)
{
	$ObjectiveHUD::BlinkOn[%id] = false;
	ObjectiveHUD::onLoaded();
	$schedule[ObjectiveHUD::Blink, %id]++;
	schedule("ObjectiveHUD::Blink("@$schedule[ObjectiveHUD::Blink,%id]@","@%id@",0);", 0.3);
}

// When an objective has been updated, update HUD plus blink the name.
function ObjectiveHUD::onUpdate(%client, %id, %status, %type)
{
	$ObjectiveHUD::BlinkOn[%id] = false;
	ObjectiveHUD::UpdateLine(%id);
	$schedule[ObjectiveHUD::Blink, %id]++;
	schedule("ObjectiveHUD::Blink("@$schedule[ObjectiveHUD::Blink,%id]@","@%id@",0);", 0.3);
}

// Blink the objective name on and off.
function ObjectiveHUD::Blink(%schedule,%id,%i)
{
	// If another update occurs, we stop the reschedule loop from the previous update here.
	if(%schedule != $schedule[ObjectiveHUD::Blink, %id]) return;

	if(%i > 10)
		$ObjectiveHUD::BlinkOn[%id] = false;
	else
	{
		$ObjectiveHUD::BlinkOn[%id] = !$ObjectiveHUD::BlinkOn[%id];
		schedule("ObjectiveHUD::Blink("@%schedule@","@%id@","@%i+1@");", 0.3);
	}

	ObjectiveHUD::UpdateLine(%id);
}

// Objective list has been reset.
function ObjectiveHUD::onReset()
{
	// Reset variables.
	$ObjectiveHUD::Last = "";
	deleteVariables("$ObjectiveHUD::BlinkOn*");
	deleteVariables("$schedule*"); // Break all schedule loops.

	// Update HUD.
	ObjectiveHUD::Setup();
}

// Create the HUD.
if(!HUD::Exists(ObjectiveHUD))
	HUD::NewFrame(ObjectiveHUD, "", $CrunchyPref::ObjectiveHUDPos);

// Setup HUD on connect or mission change.
function ObjectiveHUD::Setup()
{
	ObjectiveHUD::Resize();

	if($CrunchyPref::ObjectiveHUDDisplay == onLoaded)
		HUD::Display(ObjectiveHUD, false);
	else
		HUD::Display(ObjectiveHUD, $CrunchyPref::ObjectiveHUDDisplay);
}
ObjectiveHUD::Setup();

//
// NewOpts
//

// Control radio buttons.
function ObjectiveHUD::radioDisplay(%radio)
{
	$CrunchyPref::ObjectiveHUDDisplay = %radio;

	Control::setValue("ObjHUD::radioDisplaytrue", FALSE);
	Control::setValue("ObjHUD::radioDisplayfalse", FALSE);
	Control::setValue("ObjHUD::radioDisplayonLoaded", FALSE);

	Control::setValue("ObjHUD::radioDisplay"@%radio, TRUE);
}

// Setup options page
function ObjectiveHUD::onOpen()
{
	if($CrunchyPref::ObjectiveHUDDisplay == true)
		Control::setValue("ObjHUD::radioDisplaytrue", TRUE);
	else
		Control::setValue("ObjHUD::radioDisplaytrue", FALSE);
	if($CrunchyPref::ObjectiveHUDDisplay == false)
		Control::setValue("ObjHUD::radioDisplayfalse", TRUE);
	else
		Control::setValue("ObjHUD::radioDisplayfalse", FALSE);
	if($CrunchyPref::ObjectiveHUDDisplay == onLoaded)
		Control::setValue("ObjHUD::radioDisplayonLoaded", TRUE);
	else
		Control::setValue("ObjHUD::radioDisplayonLoaded", FALSE);
}

// Update preferences and HUD if needed.
function ObjectiveHUD::onClose()
{
	export("$CrunchyPref::*", "config\\CrunchyPrefs.cs", false);
	ObjectiveHUD::Resize();
}

// Run when Defaults button is pressed.
function ObjectiveHUD::onbuttonDefaults()
{
	ObjectiveHUD::SetDefaults();
	ObjectiveHUD::onOpen();
}

// Options page for Zear's NewOpts, <grin>
NewOpts::register("Objective HUD",
				  "config\\Crunchy\\gui\\ObjectiveHUDoptions.gui",
				  "ObjectiveHUD::onOpen();",
				  "ObjectiveHUD::onClose();",
				  TRUE);
Include("Crunchy\\ObjectiveHUDHelp.cs");


} // Presto Pack check.
