// -- ObjectiveTrak -- v2.0 beta 7 --------------------------------------------------------------------------
// Written by |HH|Crunchy (Jonathan Slark)
// Web page: http://www.crunchy.oaktree.co.uk/scripting/
//
// *** Requires the Presto Pack v0.93 or later ***  http://www.planetstarsiege.com/presto/
//
// Scripting tool that tracks the status of the objectives in missions.  The script currently tracks
// objectives from CTF, C&H, D&D and F&R mission types.  Note however you can get these types of objective in
// other mission types, the tracking isn't dependant on mission type.  For instance C&H (tower) type
// objectives can be found scatered through all the mission types.  In addition the C&H traking actually
// works in multi-team missions!  A caveat is that Presto's team name tracking only works for two teams.
// Take a look at the new events and functions ObjectiveTrak defines listed below.  For an example of the use
// of the script check out my Objective HUD.
//
// The script will learn about new objectives from the messages sent to the client about them and will then
// save  this information so that the next time you play that mission it will then know about the objectives
// right from the start of the level!  There is a pack of files available that has mission
// information, usuable in a script, for all the missions that come with Tribes.
//
// The current implemetation assigns each objective unique ID's.  The ID's are intergers counting up from
// zero to one less than the number returned by the Objective::Number() function (see below).  When Presto
// has finished his implementation of lists then I may update this script to take advantage of this.
//
// The script has the nice side effect of being able to help TeamTrak learn the names of the teams.  In a C&H
// type mission for example this is the only way to find out the team names.
//
// This script uses plugins to do the work with different objective types.  A scripter could write a plugin
// for a new objective type.  The plugin would be an extra script that the end user executes in addition to
// this script.  See the bottom part of the script for more detail.
//
// Example:
//
//	To check the status of all the objectives you could use a simple loop to run through them:
//
//		for(%id = 0; id < Objective::Number(); id++)
//		{
//			%status = Objective::GetStatus(%id);
//			%type = Objective::GetType(%id);
//			if(%type == $Objective::CandH)
//			{
//				if(%status == $Objective::Enemy)
//					(do something useful regarding enemy having the objective)
//				else if...
//			}
//		}
//
//
// Changes in v2.0 beta 7:
//  = Moved eventObjectiveLoaded so that scripts can avoid having to use eventObjectiveMadeNeutral as well.  
//	  These events were very close together and so could cause problems.
//
// Changes in v2.0 beta 6:
//	+ Added support for scores in the mission info files.
//	+ Added score code to Objective::Set() and the plugins.
//	+ Added support functions Objective::GetTeamScore(), Objective::GetScore(), Objective::GetDeltaScore(), 
//	  Objective::GetTime(), Objective::GetTimeScore(), Objective::GetWinScore().
//  = Changed F&R tracking to work with Tribes v1.6's new 180sec timelimit for conveying a flag.
//	= Objective::InsertTeamName() returns the team name (if known) instead of "Enemy ..." when observing.
//
// Changes in v2.0 beta 5:
//	= Client message parser didn't check to see if the message was from the server. D'oh :)
//
// Changes in v2.0 beta 4:
//	+ Added new events:
//		eventObjectiveTaken
//		eventObjectiveLost
//		eventObjectiveFlagTaken
//		eventObjectiveFlagDropped
//		eventObjectiveFlagReturned
//		eventObjectiveFlagCaptured
//		eventObjectiveFlagHolds
//  + CTF flags support!
//  + Added support function Objective::InsertTeamSymbol().
//	= Support function Objective::GetId() optimised.  Now executes in constant rather than linear time.
//  = Flag (F&R and CTF) carry state was being stored fixed, now changes correctly if you change teams.
//
// Changes in v2.0 beta 3:
//  = Fixed another CTF / F&R flag clash.
//	= Fixed bug where abbreviated names for new objectives weren't being stripped.
//	- Moved abbreviations to the MissionInfo Pack.
//
// Changes in v2.0 beta 2:
//	+ Added support function Objective::GetNameAbbreviated(%id).  Abbreviations are stored in a table for
//	  efficiency.
//  + Had to add toggle key to distinguish between CTF and F&R flags, $ServerMissionType isn't always set :/.
//
// Changes in v2.0 beta:
//	+ Compelete rewrite.  Generalised the script so that it could track any objective.
//  + Beta version released to get bug reports as testing is difficult.
//  + Used the logic and messages from scripts.vol to derive the messages and parsing functions.
//
// Changes in v1.0.1:
//	= Fixed bug where Objective::GetNameStripped(%id) would return a null value if there was no "the" at
//	  the start of the name (don't rename variables ;).
//
// What happend to v1.0?
//	+ I first wrote the script to track C&H mission objectives but Presto pointed out it could be
//	  generalised to track any objective.  Thanks for the idea Presto :)
//
//
// To do:
//  + Multi-team names?
//  + Multi-team CTF?
//
//
// New events:
//
//	eventObjectiveReset
//
//		Objectives variables have just been reset between missions or when first connected.  There will be no
//		objectives being tracked.  We won't know about them until they are loaded or discovered about mid
//		game.
//
//	eventObjectiveLoaded(%num)
//
//		Called when the objectives are loaded off disk, the number loaded is returned.  If they can't be
//		loaded the objectives will be added when they are taken or lost.  After each mission any new
//		objectives will be saved to the config directory so that the next time you play that level the
//		script will know about the objectives.  There is a file pack available that contains objective
//		information for all the base missions.  These are used if found in the config\Mission directory, any
//		files in the config directory will take precedence however.  If only a partial list was saved out
//		then the script will still learn about new objectives when they are claimed.
//
//	eventObjectiveMadeNeutral()
//
//		On a mission change or if you join a server just before the match starts the script knows that all
//		the objectives are neutral.  It sets them neutral and then triggers this event.
//
//	eventObjectiveNew(%client, %id, %status, %type)
//
//		Just found about a new objective.  The event returns the client ID of the player that changed the
//		state of the objective, the ID of the objective, the status of the objective and the type of the
//		objective.  The client ID is 0 if it was the server that changed the status of the objective, for
//		instance returning an F&R flag after it was dropped and no one picked it up.
//
//	eventObjectiveUpdated(%client, %id, %status, %type)
//
//		An objective already known about has changed state in some way.  The event returns the client ID of
//		the player that took the objective, the ID of the objective, the new status of the objective and the
//		type of the objective.  The client ID is 0 if it was the server that changed the status of the
//		objective.
//
//
// Events more specific than eventObjectiveUpdated:
//
//	Use the following events instead of eventObjectiveUpdated if they provide what you need.  This ensures
//	your script has to do little as possible.
//
//	eventObjectiveTaken(%client, %id, %team, %type)
//
//		Whenever an objective is taken by your team this is triggered.  It returns the client that took the
//		objective and the objective name.  Objectives that can be taken are C&H towers and D&D objectives.
//		In the case of D&D objectives "take" means your team destroyed the enemy objective.
//
//	eventObjectiveLost(%client, %id, %team, %type)
//
//		Whenever an objective that your team was holding is lost this is triggered.  It returns the client
//		that took the objective, the objective name and the team that took it.  Objectives that can be lost
//		are C&H towers and D&D objectives.  In the case of D&D objectives "lost" means your team failed to
//		defend one of your objectives.
//
//	eventObjectiveFlagTaken(%client, %id, %team, %type)
//
//		This event is triggered when a flag has been taken.  It returns the client that is now carrying the
//		flag, the flag's ID and the team number.  The type can be a CTF flag or F&R flag.  In CTF missions
//		the team is that of the flag, for F&R flags it is the team of the taker.
//
//	eventObjectiveFlagDropped(%client, %id, %team, %type)
//
//		A flag was dropped in the field.  The client that dropped the flag is returned.  Also the flag ID
//		and team are returned.  The type can be a CTF flag or F&R flag.  In CTF missions the team is that of
//		the flag, for F&R flags it is the team of the dropper.
//
//	eventObjectiveFlagReturned(%client, %id, %team, %type)
//
//		A flag was returned to it's initial position.  For F&R flags this only happens when a player left the
//		mission area whilst carrying it or the flag was dropped in the field for a certain length of time.
//		For CTF flags this can be either when a player left the mission area or an enemy returned it or the
//		flag was in the field for a certain length of time.  The type can be a CTF flag or F&R flag.  In CTF
//		missions the team is that of the flag.  For F&R flags it is the team of the player that left the
//		mission area whilst carrying the flag, if that is what happened to the flag.
//
//	eventObjectiveFlagCaptured(%client, %id, %team, %type)
//
//		This teams flag was captured by this client.  Only applies to CTF flags.
//
//	eventObjectiveFlagHolds, %client, %id, %team, %type)
//
//		This team now holds this flag in their base.  The client number of the player that conveyed the flag
//		to their base is also returned.  Only applies to F&R flags.
//
//
// New functions:
//
//	Objective::GetStatus(%id);
//
//		Returns the status of the objective that is associated with the ID.  Possible values returned
//		is dependant on the objective type.
//
//			CTF:
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Enemy
//					The flag is in the enemy base.
//				$Objective::Friendly
//					The flag is in your base.
//				$Objective::FriendlyCarry
//					Either you or a team mate is carrying the flag.
//				$Objective::EnemyCarry
//					An enemy is carrying the flag.
//				$Objective::Dropped
//					The flag has been dropped in the field
//
//			C&H:
//				$Objective::Neutral
//					The match has just started, no one has control of the tower.
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Enemy
//					An enemy team has control of the tower.
//				$Objective::Friendly
//					Your team has control of the tower.
//
//			D&D:
//				$Objective::Neutral
//					The match has just started, the objective has not been destroyed.
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Destroyed
//					The objective has been destroyed.
//
//			F&R:
//				$Objective::Neutral
//					The match has just started, the flag is at its starting postition.
//				$Objective::Unknown
//					We joined mid-game, we don't know what the status is.
//				$Objective::Enemy
//					The flag is in the enemy base.
//				$Objective::Friendly
//					The flag is in your base.
//				$Objective::FriendlyCarry
//					Either you or a team mate is carrying the flag.
//				$Objective::EnemyCarry
//					An enemy is carrying the flag.
//				$Objective::Dropped
//					The flag has been dropped in the field
//
//  Objective::GetRawStatus(%id)
//
//		Similar to above but doesn't interpret the status for you.  This is the internal representation of
//		the state.  Use the above function in preference.
//
//	Objective::GetName(%id);
//
//		Get the name of the objective associated with the ID.
//
//	Objective::GetNameStripped(%id);
//
//		Get the name of the objective associated with the ID but strip off a "the" if it exists.  For
//		example if the name of the an objective was "the Central Citadel" this function would return
//		"Central Citadel".
//
// Objective::GetNameAbbreviated(%id)
//
//		Get the name of the objective associated with the ID but abbreviate the name, strips it as above as
//		well.  This makes the objective name a lot shorter in many cases.  For instance "Secondary Generator"
//		is abbreviated to "2nd Gen".  See the file Abbreviations.cs in the MissionInfo Pack for more
//		information.
//
//	Objective::Number();
//
//		Get the number of objectives we currently know about, this is the number loaded from disk plus
//		those we have found out about since.
//
//	Objective::GetId(%objectiveName);
//
//		Get the ID of an objective from its name.  Note that this has to be the entire name, "the Central
//		Citadel" in the example above.  Returns -1 if the objective doesn't exist.
//
//	Objective::GetType(%id);
//
//		Returns the type of this objective.  Values can be:
//			$Objective::CandH - Capture and Hold (tower)
//			$Objective::DandD - Defend and Destroy
//			$Objective::FandR - Find and Retrieve (flag)
//			$Objective::CTF - Capture the Flag (flag)
//
//	Objective::GetTime(%id)
//
//		Get the time in seconds since the map start (getSimTime()) that the objective last changed state.
//
//	Objective::GetTimeScore(%id, %team)
//
//		Get the score linked to time for this objective.  This is only valid for objectives the team 
//		currently holds.
//
//	Objective::GetScore(%id)
//
//		Get the fixed amount score associated with capturing / destroying / holding an objective.
//
//	Objective::GetTeamScore(%team)
//
//		Get the score of this team as a float.
//
//	Objective::GetWinScore()
//
//		Get the score needed for a team to win this mission.
//
//	Objective::GetDeltaScore(%id)
//
//		Get the score per minute associated with this objective.
//
//	Objective::GetClient(%id)
//
//		Get the client involved with the last change of state.  In the case of someone carrying the flag this
//		would that players client number.  Otherwise it's whoever claimed, captured etc the objective.  If
//		there was no played involved in the change then the value is 0.  ie It was the server that made the
//		change.  This could happen in the case of returning a flag after a timeout for instance.
//
//	Objective::InsertTeamName(%msg)
//
//		In certain objective names we have to remember the team number involved, it is stored as a wild card.
//		For instance in a D&D type game: "%0 Main Generator".  This function searches for any wildcards and
//		inserts either "your" or "enemy" into it's place.  It even checks to see if word is at the start of
//		the sentance and so will capitalise the message.  In the example above: "Your Main Generator", if you
//		are on team 0.  This could've put the actual team name into the message but this makes the names very
//		long and for a user it is easier to work out who's objective it is.
//
//	Objective::InsertTeamSymbol(%msg)
//
//		Same as Objective::InsertTeamName() but inserts a red or a green player symbol into the text instead.
//		A red player indicates an enemy objective, green means friendly objective.
//
//	Objective::PrintAll()
//
//		Useful for debuging, this function prints to the console the entire list of objectives currently
//		known about, with all their information.
//
//
// -- Preferences --
//
// If $ServerMissionType has not been set on a server the script will not know the difference between CTF and
// F&R flags.  You can set which the script should be tracking and it will fall back on this if the variable
// hasn't been set on the server.
$CrunchyPref::FlagTogKey = "alt a";
//

//
// -- Header --
//

Include("Presto\\Event.cs");
Include("Crunchy\\Events.cs");
Include("Presto\\TeamTrak.cs");
Include("Presto\\Match.cs");

Event::Attach(eventChangeMission, "Objective::Save();Objective::Reset();$MidGame = false;");
Event::Attach(eventConnectionAccepted, "Objective::Reset();$MidGame = true;");
Event::Attach(eventClientMessage, Objective::Parse);
Event::Attach(eventMissionInfo, Objective::Load);

// Interface with Team Trak
Event::Attach(eventFlagTaken, Objective::FlagTaken);
Event::Attach(eventFlagDropped, Objective::FlagDropped);
Event::Attach(eventFlagCaptured, Objective::FlagCaptured);
Event::Attach(eventFlagReturned, Objective::FlagReturned);

//
// -- Begin code --
//

// Toggle to set mode if $ServerMissionType has not been set.
bindKey(play, $CrunchyPref::FlagTogKey, "Objective::ToggleFlagType();");
$Objective::ToggleFlag = ctf; // Defaults to CTF flags.
function Objective::ToggleFlagType()
{
	if($Objective::ToggleFlag == ctf)
	{
		$Objective::ToggleFlag = fandr;
		%msg = "F&R";
	}
	else
	{
		$Objective::ToggleFlag = ctf;
		%msg = "CTF";
	}

	remoteBP(2048, "<f0>	Tracking <f2>"@%msg@"<f0> flags",5);
}

// Function to parse the incoming messages.  Compares the message to a list we have created and triggers
// an action function if we have a match.
function Objective::Parse(%client, %msg)
{
	if(%client) return;

	// Watch for a match start and make the objectives neutral in that case.
	if(Match::String(%msg, "Match starts in * seconds.") || %msg=="Match started.")
	{
		if($Objective::Loaded && $MidGame)
			Objective::MakeNeutral();

		$MidGame = false;
	}
	else
	{
		// Loop through all the messages.
		for(%i = 0; %i < $ObjMsg::Num; %i++)
		{
			// On a match call the associated action function.
			if(Match::ParamString(%msg, $ObjMsg::[%i]))
			{
				%p = Match::Result(p);
				%o = Match::Result(o);
				%t = Match::Result(t);
				%action = $ObjMsg::[%i, action];
				eval("Objective::Action_"@%action@"(%p, %o, %t);");

				return true;
			}
		}
	}

	return true;
}

// Update value of an objective.
function Objective::Set(%client, %o, %status, %type, %friendlyScore, %enemyScore)
{
	%id = Objective::GetId(%o);
	if(%id == -1)
	{
		// Objective doesn't exist so create a new one.
		%id = $ObjData::Num;
		$ObjData::Num++;
		%new = true;
		$Objective::New = true;
	}
	else
		%new = false;

	// Set scores
	%score = Objective::GetScore(%id);
	if(%client != 0)
	{
		%team = Client::getTeam(%client);
		%enemy = Team::Enemy(%client);
	}
	else
	{
		%team = %status;
		%enemy = 1-%team;
	}
	$ObjData::Score[%team] += %score*%friendlyScore;
	$ObjData::Score[%enemy] += %score*%enemyScore;
	$ObjData::Score[%enemy] += Objective::GetTimeScore(%id, Team::Enemy(%client));

	// Update the objective.
	$ObjData::[%id, client] = %client;
	$ObjData::[%id, name] = %o;
	$ObjData::[%id, status] = %status;
	$ObjData::[%id, type] = %type;
	$ObjData::[%id, time] = getSimTime();
	$ObjData::[%o] = %id;

	// Interpret the status.
	%status = eval("Objective::Interpret_"@%type@"(%id, $ObjData::[%id, status]);");
	// Triggers.
	if(%new)
	{
		$ObjData::[%id, abrv] = Objective::Abbreviate(Objective::GetNameStripped(%id));
		Event::Trigger(eventObjectiveNew, %client, %id, %status, %type);
	}
	else
		Event::Trigger(eventObjectiveUpdated, %client, %id, %status, %type);

	return %id;
}

// Load the objective list off disk if available.
function Objective::Load(%server, %missionName, %missionType)
{
	function Objective::Init()
	{
		$Objective::Loaded = true;
		Objective::MakeAbbreviations();
		Objective::StoreIDs();
		if(!$MidGame) Objective::MakeNeutral();
		Event::Trigger(eventObjectiveLoaded, Objective::Number());
	}

	%file = %missionName@".cs";
	if(isFile("config\\"@%file))
	{
		exec(%file);
		Objective::Init();
	}
	else if(isFile("config\\Missions\\"@%file))
	{
		exec("Missions\\"@%file);
		Objective::Init();
	}
}

// Make ID table for quick retrieval.
function Objective::StoreIDs()
{
	for(%id = 0; %id < Objective::Number(); %id++)
		$ObjData::[Objective::GetName(%id)] = %id;
}

// Save out to disk the objective list we created.
function Objective::Save()
{
	if(!$Objective::New) return;

	%file = "config\\"@$ServerMission@".cs";
	export("$ObjData::Num", %file, false);
	export("$ObjData::*_name", %file, true);
	export("$ObjData::*_type", %file, true);
	export("$ObjData::*_score", %file, true);
	export("$ObjData::WinScore", %file, true);
	if($ServerMissionType != "")
	{
		$ObjData::MissionType = $ServerMissionType;
		export("$ObjData::MissionType", %file, true);
	}
}

// If we are starting a mission where we know about some of the objectives we know they are neutral.
function Objective::MakeNeutral()
{
	for(%id = 0; %id < Objective::Number(); %id++)
		$ObjData::[%id, status] = $Objective::Neutral;

	Event::Trigger(eventObjectiveMadeNeutral);
}

// Make up the abbreviations once so we don't have to do this expensive operation all the time.
function Objective::MakeAbbreviations()
{
	for(%id = 0; %id < Objective::Number(); %id++)
		$ObjData::[%id, abrv] = Objective::Abbreviate(Objective::GetNameStripped(%id));
}

// Get the ID from the name of an objective.
function Objective::GetId(%o)
{
	if($ObjData::[%o] == "")  // Compatiblity with Objective::GetId() from v2.0 beta 3
		return -1;
	else
		return $ObjData::[%o];
}

// Inefficient, only use if you HAVE too :)
// This is used once, and then in a rare case.
function Objective::GetCarry(%client)
{
	for(%id = 0; %id < Objective::Number(); %id++)
	{
		if(Objective::GetStatus(%id) == $Objective::FriendlyCarry
		|| Objective::GetStatus(%id) == $Objective::EnemyCarry)
		{
			if(Objective::GetClient(%id) == %client)
				return %id;
		}
	}

	return "";
}

// Work out client from the message.
function Objective::GetClientFromMsg(%p)
{
	if(%p == "You")
		return getManagerId();
	else
		return getClientByName(%p);
}

// Number of those we know about up to now.
function Objective::Number()
{
	return $ObjData::Num;
}

// Get the name of an objective.
function Objective::GetName(%id)
{
	return $ObjData::[%id, name];
}

// Strip the "the" from a name.
function Objective::GetNameStripped(%id)
{
	%o = Objective::GetName(%id);
	if(Match::ParamString(%o, "the %o"))
		return Match::Result(o);
	else
		return %o;
}

// Get the abbreviated version of the objective name.
function Objective::GetNameAbbreviated(%id)
{
	return $ObjData::[%id, abrv];
}

// Get the current status of an objective in terms of the $Objective::* variables.
function Objective::GetStatus(%id)
{
	%type = Objective::GetType(%id);
	%status = eval("Objective::Interpret_"@%type@"(%id, $ObjData::[%id, status]);");
}

// Get current status but don't interpret.
function Objective::GetRawStatus(%id)
{
	return $ObjData::[%id, status];
}

// Get the type of the objective.
function Objective::GetType(%id)
{
	return $ObjData::[%id, type];
}

// Get the time of the last update on the objective
function Objective::GetTime(%id)
{
	return $ObjData::[%id, time];
}

// Score associated with this objective for this team since the last update.
function Objective::GetTimeScore(%id, %team)
{
	%deltaScore = Objective::GetDeltaScore(%id);
	if((%timeUpdate = Objective::GetTime(%id)) != "" && Objective::GetRawStatus(%id) == %team)
		return (getSimTime()-%timeUpdate)*%deltaScore/60;
	else
		return 0;
}

// Get the score of this team.
function Objective::GetTeamScore(%team)
{
	%baseScore = $ObjData::Score[%team];

	%timeScore = 0;
	for(%id = 0; %id < Objective::Number(); %id++)
		%timeScore += Objective::GetTimeScore(%id, %team);

	return %baseScore + %timeScore;
}

// Get the point value of an objective.
function Objective::GetScore(%id)
{
	%score = $ObjData::[%id, score];
	if(%score == "")
		return 0;
	else
		return getWord(%score, 0);
}

// Get the score required for a team to win outright.
function Objective::GetWinScore()
{
	return $ObjData::WinScore;
}

// Get the points per minute of an objective.
function Objective::GetDeltaScore(%id)
{
	%score = $ObjData::[%id, score];
	if(%score == "")
		return 0;
	else
		return getWord(%score, 1);
}

// Get the client number involved in the last status change.
function Objective::GetClient(%id)
{
	return $ObjData::[%id, client];
}

// Clear the list between missions.
function Objective::Reset()
{
	deleteVariables("$ObjData::*");
	Objective::ResetTeamScores();
	$ObjData::Num = 0;
	$Objective::Loaded = false;
	$Objective::New = false;

	Event::Trigger(eventObjectiveReset);
}

function Objective::ResetTeamScores()
{
	// Maximum of 8 teams (from scripts.vol)
	for(%i = 0; %i < 8; %i++)
		$ObjData::Score[%i] = 0;
}

// Useful for debuging, prints the name and status of an objective.
function Objective::Print(%id)
{
	echo("Player: ", Client::getName(Objective::GetClient(%id)));
	echo("Name: ", Objective::GetName(%id));
	echo("Status: ", Objective::GetStatus(%id));
	echo("Type: ", Objective::GetType(%id));
}

// Print the names and status of all the objectives we know about.
function Objective::PrintAll()
{
	echo("Friendly Team: ", Team::GetName(Team::Friendly()));
	echo("Enemy Team: ", Team::GetName(Team::Enemy()));
	echo("Scores: ", Objective::GetTeamScore(0), "-", Objective::GetTeamScore(1));
	for(%id = 0; %id < Objective::Number(); %id++)
		Objective::Print(%id);
}

// Setup message database.
$ObjMsg::Num = 0;
function Objective::Message(%msg, %action, %type)
{
	%i = $ObjMsg::Num;
	$ObjMsg::[%i] = %msg;
	$ObjMsg::[%i, action] = %action;
	$ObjMsg::Num++;
}

// Setup abreviations.
function Objective::AbbreviationReset()
{
	$ObjAbrv::Num = 0;
}
function Objective::Abbreviation(%name, %len, %abrv)
{
	%i = $ObjAbrv::Num;
	$ObjAbrv::[%i] = %name;
	$ObjAbrv::[%i, len] = %len;
	$ObjAbrv::[%i, abrv] = %abrv;
	$ObjAbrv::Num++;
}

// Abbreviate objective names.
function Objective::Abbreviate(%msg)
{
	%abrv = true;
	while(%abrv)
	{
		// If this is still false at end of loop we didn't find any more occurances so we can exit.
		%abrv = false;
		for(%i = 0; %i < $ObjAbrv::Num; %i++)
		{
			if((%pos = String::findSubStr(%msg, $ObjAbrv::[%i])) != -1)
			{
				%msg = String::getSubStr(%msg, 0, %pos) @
					   $ObjAbrv::[%i, abrv] @
					   String::getSubStr(%msg, %pos + $ObjAbrv::[%i, len], 256);
				%abrv = true;
			}
		}
	}

	return %msg;
}

// Work out the team the message was refering to.
// No longer used, but could be useful :)
function Objective::GetTeamDet(%d)
{
	if(%d == "an")
		return Team::Friendly();
	else
		return Team::Enemy();
}

// Where the name of an objective is dependant on the team, the team number is stored as "%0" etc.  When we
// display it we want to insert "Your" or "Enemy" in the play of these wildcards.
// If the player is in observer mode then the name of the team is returned (if known) instead of the normal 
// name.
function Objective::InsertTeamName(%msg)
{
	while((%pos = String::findSubStr(%msg, "%")) != -1)
	{
		%team = String::getSubStr(%msg, %pos+1, 1);  // Assumes not more than 10 teams :)

		if(Team::Friendly() == -1 && (%name = Team::GetName(%team)) != "")
			return %name;
		else if(%team == Team::Friendly())
		{
			if(%pos == 0)
				%teamName = "Your";
			else
				%teamName = "your";
		}
		else
		{
			if(%pos == 0)
				%teamName = "Enemy";
			else
				%teamName = "enemy";
		}
		%msg = String::getSubStr(%msg, 0, %pos) @ %teamName @ String::getSubStr(%msg, %pos+2, 256);
	}

	return %msg;
}

// Add team icon to objectives.
function Objective::InsertTeamSymbol(%msg)
{
	if((%pos = String::findSubStr(%msg, "%")) != -1)
	{
		%team = String::getSubStr(%msg, %pos+1, 1);  // Theres an 8 team limit in scripts.vol
		if(%team == Team::Friendly())
		{
			%teamName = "<B-4,3:M_Player_green.bmp>";
		}
		else
		{
			%teamName = "<B-4,2:M_Player_red.bmp>";
		}
		%msg = String::getSubStr(%msg, 0, %pos) @ %teamName @ String::getSubStr(%msg, %pos+2, 256);
	}

	return %msg;
}

//
// The plugins
// ===========
//
// This is the cool bit :).  Here we add the messages that we are going to parse.  Each message has an
// associated action function.  The message provides the syntax and the action function gives the semantics.
// In the action function we set the new status of the objective and do any processing required.  These
// functions don't have to be stored in this file.  As long as a script containing more plugins is executed
// alongside this script they will get used.
//
//
// Status values
// -------------
//
// The status of the objective can be one of the following things, the actual meaning depends on the
// objective in question.  All objectives can be unknown or neutral however.  The status is returned by the
// Objective::GetStatus(%id).  Note that the internal representation of the state can be different, these
// values are the ones handed to the end user.  For instance store something team dependant using team
// numbers and return the status $Objective::Enemy if the team number stored isn't the same as the players
// team number.
//
$Objective::Enemy			= enemy;
$Objective::Friendly		= friendly;
$Objective::Destroyed		= destroyed;
$Objective::Neutral			= neutral;
$Objective::Carry			= carry; // This will be interpreted as one of the following two.
$Objective::FriendlyCarry	= carryFriendly;
$Objective::EnemyCarry		= carryEnemy;
$Objective::Dropped			= dropped;
$Objective::Unknown			= unknown;
//
//
// Objective types
// ---------------
//
// The types currently supported.  This is returned by the Objective::GetType(%id) function.
//
$Objective::CandH			= candh;
$Objective::DandD			= dandd;
$Objective::FandR			= fandr;
$Objective::CTF				= ctf;
//
//
// Action functions
// ----------------
//
// Each message has an action function associated with it.  The function is passed three variables, see the
// messages section.  Messages could share the same action function.  In these functions you can do whatever
// you need to, this script just provides a way to keep track of the objectives.  For instance you could
// trigger extra events.
//
// The minimum the action function should provide is running the following function with appropriate values:
//		%id = Objective::Set(%client, %o, %status, %type, %friendlyScore, %enemyScore);
// 			This function either creates or updates the objective, it returns the ID of the objective.
// Note that this function doesn't have to be called straight away, it is sometimes nessary to delay the call
// until we have more information provided by another message.  Use two or more action functions in tandom
// for this.  The %friendlyScore and %enemyScore indicate how the scores should be updated.  0 means no 
// change, 1 means add a fixed score unit and -1 means take away a fixed score unit.  A fixed score unit is 
// score added on for events such as a flag cap, as apposed to a score linked to time.  The friendly is 
// either the team of the client or if the client is 0 its taken to be %status.
//
// Objective::Action_"action"(%p, %o, %t);
//		The action function is always named Objective::Action_ followed by the name of the action.  See the
//		message section.
//
// claimed: an objective has just been claimed.
function Objective::Action_claimed(%p, %o, %t)
{
	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);
	%id = Objective::Set(%client, %o, %team, candh, 1, 0);

	$ObjData::Team = %team;

	if(%team == Team::Friendly())
		Event::Trigger(eventObjectiveTaken, %client, %id, %team, candh);
	else
		Event::Trigger(eventObjectiveLost, %client, %id, %team, candh);
}
//
// captured: an objective has been captured.
function Objective::Action_captured(%p, %o, %t)
{
	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);
	if($ServerMissionType != "Multiple Team") Team::SetName(Team::Enemy(%client), %t);
	%id = Objective::Set(%client, %o, %team, candh, 1, -1);


	$ObjData::Team = %team;

	if(%team == Team::Friendly())
		Event::Trigger(eventObjectiveTaken, %client, %id, candh);
	else
		Event::Trigger(eventObjectiveLost, %client, %id, %team, candh);
}
//
// destroy: Just destroyed an objective.
function Objective::Action_destroy(%p, %o, %t)
{
	%team = $ObjData::Team;
	%client = $ObjData::Client;
	Team::SetName(%team, %t);

	if(%team == Team::Friendly(%client))
		%id = Objective::Set(%client, "%"@%team@" "@%o, $Objective::Destroyed, dandd, 0, 1);
	else
		%id = Objective::Set(%client, "%"@%team@" "@%o, $Objective::Destroyed, dandd, 1, 0);

	if(%team == Team::Friendly())
		Event::Trigger(eventObjectiveTaken, %client, %id, dandd);
	else
		Event::Trigger(eventObjectiveLost, %client, %id, %team, dandd);
}
//
// teamName: we just found out about a team name.
function Objective::Action_teamName(%p, %o, %t)
{
	if(Match::ParamString(%t,"%t team"))
		%t = Match::Result(t);
	Team::SetName($ObjData::Team, %t);
}
//
// playerEnemy: get an enemy player name we need for later on.
function Objective::Action_playerEnemy(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	$ObjData::Team = Team::Enemy($ObjData::Client);
}
//
// playerFriendly: get a friendly player name we need for later on.
function Objective::Action_playerFriendly(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	$ObjData::Team = Client::getTeam($ObjData::Client);
}
//
// carry: player has picked up a flag and is carrying it.
function Objective::Action_carry(%p, %o, %t)
{
	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);
	$ObjData::Team = %team;

	if(Match::ParamString(%o, "%o from the %t team"))
	{
		%o = Match::Result(o);
		%t = Match::Result(t);
		Team::SetName(Team::Enemy(%client), %t);
	}

	%enemyScore = 0;
	if((%id = Objective::GetId(%o)) != -1)
	{
		if(Objective::GetStatus(%id) == $Objective::Enemy)
		{
			%enemyScore = -1;
			//ObjData::[%id, washeld] = Objective::GetRawStatus(%id);
		}
	}
	%id = Objective::Set(%client, %o, $Objective::Carry, fandr, 0, %enemyScore);

	Event::Trigger(eventObjectiveFlagTaken, %client, %id, %team, fandr);
}
//
// dropped: flag has been dropped in the field.
function Objective::Action_dropped(%p, %o, %t)
{
	// Not implemented yet (this stops spurious objectives being created):
	if($ServerMissionType == "Capture the Flag" || $ServerMissionType == "Multiple Team") return;
	if($Objective::ToggleFlag == ctf && $ServerMissionType != "Find and Retrieve") return;

	%client = Objective::GetClientFromMsg(%p);
	%team = Client::getTeam(%client);

	%id = Objective::Set(%client, %o, $Objective::Dropped, fandr, 0, 0);

	Event::Trigger(eventObjectiveFlagDropped, %client, %id, %team, fandr);
}
//
// conveyed: flag has been conveyed to base.
function Objective::Action_conveyed(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	$ObjData::Team = Client::getTeam($ObjData::Client);
}
//
// neutral: flag was returned to neutral stand.
function Objective::Action_neutral(%p, %o, %t)
{
	if(%p != "")
		%client = Objective::GetClientFromMsg(%p);
	else
		%client = 0;

	%team = Client::getTeam(%client);

	%id = Objective::Set(%client, %o, $Objective::Neutral, fandr, 0, 0);

	Event::Trigger(eventObjectiveFlagNeutral, %client, %id, %team, fandr);
}
//
// returned: flag was returned to base.
function Objective::Action_returned(%p, %o, %t)
{
	if($ServerMissionType == "Capture the Flag" || $ServerMissionType == "Multiple Team") return;
	if($Objective::ToggleFlag == ctf && $ServerMissionType != "Find and Retrieve") return;

	%client = 0;

	if(%o == "")
	{
		%o = $ObjData::Name;
		%client = $ObjData::Client;
		%team = Team::Enemy(%client);
		%id = Objective::Set(%client, %o, %team, fandr, 0, 1);
	}
	else
	{
		if(%t == "Your")
		{
			%team = Team::Friendly();
			%id = Objective::Set(%client, %o, %team, fandr, 1, 0);
		}
		else
		{
			%team = Team::Enemy();
			%id = Objective::Set(%client, %o, %team, fandr, 1, 0);
		}
	}
}
//
//
function Objective::Action_playerName(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
}
//
// holds: a team holds one of the flags
function Objective::Action_holds(%p, %o, %t)
{
	%id = Objective::Set($ObjData::Client, %o, $ObjData::Team, fandr, 1, 0);
	if(%t != "Your")
		Team::SetName($ObjData::Team, %t); // Assuming two teams.

	Event::Trigger(eventObjectiveFlagHolds, $ObjData::Client,
											%id,
											$ObjData::Team,
											fandr);
}
//
// playerLeft: a player left the mission area whilst carrying a flag.
function Objective::Action_playerLeft(%p, %o, %t)
{
	$ObjData::Client = Objective::GetClientFromMsg(%p);
	// The message doesn't tell us the objective so we have to lookup what the player was carrying.
	$ObjData::Name = Objective::GetName(Objective::GetCarry($ObjData::Client));
	$Objective::PlayerLeft = true;
}
//
//
// Interface for Team Trak
// -----------------------
//
// Rather than duplicating the code from Presto's TeamTrak I have provided an interface to his script so that
// CTF flags can be tracked in the same way objectives of different types are.
//
function Objective::FlagTaken(%team, %client)
{
	%o = "%"@%team@" flag";
	%id = Objective::Set(%client, %o, $Objective::Carry, ctf, 0, 0);

	Event::Trigger(eventObjectiveFlagTaken, %client, %id, %team, ctf);
}
//
function Objective::FlagDropped(%team, %client)
{
	%status = $Objective::Dropped;

	%o = "%"@%team@" flag";
	%id = Objective::Set(%client, %o, %status, ctf, 0, 0);

	Event::Trigger(eventObjectiveFlagDropped, %client, %id, %team, ctf);
}
//
function Objective::FlagCaptured(%team, %client)
{
	%status = $Objective::Neutral;

	%o = "%"@%team@" flag";
	%id = Objective::Set(%client, %o, %status, ctf, 1, 0);

	Event::Trigger(eventObjectiveFlagCaptured, %client, %id, %team, ctf);
}
//
function Objective::FlagReturned(%team, %client)
{
	// Not implemented yet (this stops spurious objectives being created):
	if($ServerMissionType == "Find and Retrieve" || $ServerMissionType == "Multiple Team") return;
	if($Objective::ToggleFlag == fandr && $ServerMissionType != "Capture the Flag") return;

	%status = $Objective::Neutral;

	%o = "%"@%team@" flag";
	%id = Objective::Set(%client, %o, %status, ctf, 0, 0);

	Event::Trigger(eventObjectiveFlagReturned, %client, %id, %team, ctf);
}
//
//
// Interpreting status
// -------------------
//
// We don't necessarily pass the stored internal state to the user.  It probably needs interpretation, for
// instance if we store team numbers.  This function is called by Objective::GetStatus(%id).
//
// Objective::Interpret_"type"(%id, %status);
//		The interpret function is always named Objective::Interpret_ followed by the type of the objective.
//
function Objective::Interpret_candh(%id, %status)
{
	if(%status == Team::Friendly())
		return $Objective::Friendly;
	else if(%status == "")
		return $Objective::Unknown;
	else if(%status == $Objective::Neutral)
		return %status;
	else
		return $Objective::Enemy;  // Works with multiple teams.  Can't tell you exact enemy team though.
}
//
function Objective::Interpret_dandd(%id, %status)
{
	if(%status == "")
		return $Objective::Unknown;
	else
		return %status;
}
//
function Objective::Interpret_fandr(%id, %status)
{
	if(%status == $Objective::Carry)
	{
		if(Client::GetTeam(Objective::GetClient(%id)) == Team::Friendly())
			return $Objective::FriendlyCarry;
		else
			return $Objective::EnemyCarry;
	}
	else if(%status == Team::Friendly())
		return $Objective::Friendly;
	else if(%status == Team::Enemy())
		return $Objective::Enemy;
	else if(%status == "")
		return $Objective::Unknown;
	else
		return %status;
}
//
function Objective::Interpret_ctf(%id, %status)
{
	if(%status == $Objective::Carry)
	{
		if(Client::GetTeam(Objective::GetClient(%id)) == Team::Friendly())
			return $Objective::FriendlyCarry;
		else
			return $Objective::EnemyCarry;
	}
	else if(%status == $Objective::Neutral)
	{
		if(Objective::GetName(%id) == "%"@Team::Friendly()@" flag")
			%status = $Objective::Friendly;
		else
			%status = $Objective::Enemy;
	}
	else if(%status == "")
		return $Objective::Unknown;
	else
		return %status;
}
//
//
// Objective messages
// ------------------
//
// Here we add the actual messages to parse.
//
// Objective::Message(%msg, %action);
//		%msg: Text message.
//		%action: Function to be called if we have a match.
//
// The values that are passed to the action function:
//		%p: player name
//		%o: objective name
//		%t: a team name
//
//		%z: punctuation etc, _not_ passed to the functions.
//
// C&H
Objective::Message("%p claimed %o for the %t team!", claimed);
Objective::Message("%p captured %o from the %t team!", captured);
Objective::Message("The %t has taken %z objective.", teamName); // %z is "an" or "your".
//Objective::Message("Your team has taken an objective.", teamName);  // Can't learn anything.
//
// D&D
Objective::Message("%p destroyed a friendly objective%z", playerFriendly); // %z is "." or "!".
Objective::Message("%p destroyed an objective!", playerEnemy);
Objective::Message("%t objective %o destroyed.", destroy);
//
// F&R (updated for Tribes v1.6)
Objective::Message("%p took %o.", carry);
//Objective::Message("%p took %o from the %t team.", carry); // Covered by above msg.
//Objective::Message("Your team has %o.", carry); // Can't learn anything.
//Objective::Message("Your team lost %o.", carry); // Can't learn anything.
Objective::Message("The %t team has taken %o.", teamName);
Objective::Message("%p dropped %o!", dropped);
Objective::Message("%t team holds %o.", holds);
Objective::Message("%p conveyed %o to base.", conveyed);
//Objective::Message("Flag not in mission area.", ?); // Doesn't effect status.
Objective::Message(
	"%p didn't get %o to a flag stand in time!  It was returned to its initial position.", neutral);
Objective::Message(
	"%p didn't put %o in a flag stand in time!  It was returned to its initial position.", neutral);
Objective::Message(
	"%o was not put in a flag stand in time!  It was returned to its initial position.", neutral);
Objective::Message(
	"%p left the mission area while carrying %o!  It was returned to its initial position.", neutral);
Objective::Message("%p left the mission area while carrying the %t flag!", playerLeft);
//Objective::Message(%o was not put in a flag stand in time!", returned); // Learn more in next msg.
Objective::Message("%p didn't get %o to a flag stand in time!", playerName);
Objective::Message("%p didn't put %o in a flag stand in time!", playerName);
Objective::Message("%o was returned to %t base.", returned);
Objective::Message("The %t flag was returned to base.", returned);
//
// CTF - to do?
//Objective::Message("%p took %o! ", carry);
//Objective::Message("%p dropped %o!", dropped); // Same as F&R dropped.
//Objective::Message("Your flag was dropped in the field.", dropped);  // Can't learn anything.
//Objective::Message("%o was dropped in the field.", dropped);  // Can't learn anything.
//Objective::Message("%p returned %o!", returned);
//Objective::Message("%o was returned to base.", returned); // Same as F&R returned.
//Objective::Message("%p captured %o!", cap);
//
//
// Abbreviations - MissionInfo Pack
//
if(isFile("config\\Missions\\Abbreviations.cs"))
	Include("Missions\\Abbreviations.cs");
else
	echo("Why not get the MissionInfo Pack?");
//
